/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.remote;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.Version;
import org.opensearch.action.admin.cluster.settings.ClusterUpdateSettingsRequest;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.metadata.IndexMetadata;
import org.opensearch.cluster.metadata.Metadata;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.node.DiscoveryNodes;
import org.opensearch.cluster.routing.RoutingTable;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.settings.Settings;
import org.opensearch.index.remote.RemoteMigrationIndexMetadataUpdater;
import org.opensearch.index.remote.RemoteStoreEnums;
import org.opensearch.index.remote.RemoteStorePathStrategy;
import org.opensearch.indices.RemoteStoreSettings;
import org.opensearch.node.remotestore.RemoteStoreNodeAttribute;
import org.opensearch.node.remotestore.RemoteStoreNodeService;
import org.opensearch.node.remotestore.RemoteStorePinnedTimestampService;

public class RemoteStoreUtils {
    private static final Logger logger = LogManager.getLogger(RemoteStoreUtils.class);
    public static final int LONG_MAX_LENGTH = String.valueOf(Long.MAX_VALUE).length();
    static final char[] URL_BASE64_CHARSET = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_".toCharArray();

    public static String invertLong(long num) {
        if (num < 0L) {
            throw new IllegalArgumentException("Negative long values are not allowed");
        }
        String invertedLong = String.valueOf(Long.MAX_VALUE - num);
        char[] characterArray = new char[LONG_MAX_LENGTH - invertedLong.length()];
        Arrays.fill(characterArray, '0');
        return new String(characterArray) + invertedLong;
    }

    public static long invertLong(String str) {
        long num = Long.parseLong(str);
        if (num < 0L) {
            throw new IllegalArgumentException("Strings representing negative long values are not allowed");
        }
        return Long.MAX_VALUE - num;
    }

    public static String getSegmentName(String filename) {
        int endIdx = filename.indexOf(95, 1);
        if (endIdx == -1) {
            endIdx = filename.indexOf(46);
        }
        if (endIdx == -1) {
            throw new IllegalArgumentException("Unable to infer segment name for segment file " + filename);
        }
        return filename.substring(0, endIdx);
    }

    public static void verifyNoMultipleWriters(List<String> mdFiles, Function<String, Tuple<String, String>> fn) {
        HashMap nodesByPrimaryTermAndGen = new HashMap();
        mdFiles.forEach(mdFile -> {
            Tuple nodeIdByPrimaryTermAndGen = (Tuple)fn.apply((String)mdFile);
            if (nodeIdByPrimaryTermAndGen != null) {
                if (nodesByPrimaryTermAndGen.containsKey(nodeIdByPrimaryTermAndGen.v1()) && !((String)nodesByPrimaryTermAndGen.get(nodeIdByPrimaryTermAndGen.v1())).equals(nodeIdByPrimaryTermAndGen.v2())) {
                    throw new IllegalStateException("Multiple metadata files from different nodeshaving same primary term and generations " + (String)nodeIdByPrimaryTermAndGen.v1() + " detected ");
                }
                nodesByPrimaryTermAndGen.put((String)nodeIdByPrimaryTermAndGen.v1(), (String)nodeIdByPrimaryTermAndGen.v2());
            }
        });
    }

    static String longToUrlBase64(long value) {
        byte[] hashBytes = ByteBuffer.allocate(8).putLong(value).array();
        String base64Str = Base64.getUrlEncoder().encodeToString(hashBytes);
        return base64Str.substring(0, base64Str.length() - 1);
    }

    static long urlBase64ToLong(String base64Str) {
        byte[] hashBytes = Base64.getUrlDecoder().decode(base64Str);
        return ByteBuffer.wrap(hashBytes).getLong();
    }

    static String longToCompositeBase64AndBinaryEncoding(long value, int len) {
        if (len < 7 || len > 64) {
            throw new IllegalArgumentException("In longToCompositeBase64AndBinaryEncoding, len must be between 7 and 64 (both inclusive)");
        }
        String binaryEncoding = String.format(Locale.ROOT, "%64s", Long.toBinaryString(value)).replace(' ', '0');
        String base64Part = binaryEncoding.substring(0, 6);
        String binaryPart = binaryEncoding.substring(6, len);
        int base64DecimalValue = Integer.valueOf(base64Part, 2);
        assert (base64DecimalValue >= 0 && base64DecimalValue < 64);
        return URL_BASE64_CHARSET[base64DecimalValue] + binaryPart;
    }

    public static RemoteStorePathStrategy determineRemoteStorePathStrategy(IndexMetadata indexMetadata) {
        Map<String, String> remoteCustomData = indexMetadata.getCustomData("remote_store");
        assert (remoteCustomData == null || remoteCustomData.containsKey("path_type"));
        if (remoteCustomData != null && remoteCustomData.containsKey("path_type")) {
            RemoteStoreEnums.PathType pathType = RemoteStoreEnums.PathType.parseString(remoteCustomData.get("path_type"));
            String hashAlgoStr = remoteCustomData.get("path_hash_algorithm");
            RemoteStoreEnums.PathHashAlgorithm hashAlgorithm = Objects.nonNull(hashAlgoStr) ? RemoteStoreEnums.PathHashAlgorithm.parseString(hashAlgoStr) : null;
            return new RemoteStorePathStrategy(pathType, hashAlgorithm);
        }
        return new RemoteStorePathStrategy(RemoteStoreEnums.PathType.FIXED);
    }

    public static boolean determineTranslogMetadataEnabled(IndexMetadata indexMetadata) {
        Map<String, String> remoteCustomData = indexMetadata.getCustomData("remote_store");
        assert (remoteCustomData == null || remoteCustomData.containsKey("translog_metadata"));
        if (remoteCustomData != null && remoteCustomData.containsKey("translog_metadata")) {
            return Boolean.parseBoolean(remoteCustomData.get("translog_metadata"));
        }
        return false;
    }

    public static Map<String, String> determineRemoteStoreCustomMetadataDuringMigration(Settings clusterSettings, DiscoveryNodes discoveryNodes) {
        HashMap<String, String> remoteCustomData = new HashMap<String, String>();
        Version minNodeVersion = discoveryNodes.getMinNodeVersion();
        boolean blobStoreMetadataEnabled = false;
        boolean translogMetadata = Version.V_2_15_0.compareTo(minNodeVersion) <= 0 && RemoteStoreSettings.CLUSTER_REMOTE_STORE_TRANSLOG_METADATA.get(clusterSettings) != false && blobStoreMetadataEnabled;
        remoteCustomData.put("translog_metadata", Boolean.toString(translogMetadata));
        RemoteStoreEnums.PathType pathType = Version.V_2_15_0.compareTo(minNodeVersion) <= 0 ? RemoteStoreSettings.CLUSTER_REMOTE_STORE_PATH_TYPE_SETTING.get(clusterSettings) : RemoteStoreEnums.PathType.FIXED;
        RemoteStoreEnums.PathHashAlgorithm pathHashAlgorithm = pathType == RemoteStoreEnums.PathType.FIXED ? null : RemoteStoreSettings.CLUSTER_REMOTE_STORE_PATH_HASH_ALGORITHM_SETTING.get(clusterSettings);
        remoteCustomData.put("path_type", pathType.name());
        if (Objects.nonNull((Object)pathHashAlgorithm)) {
            remoteCustomData.put("path_hash_algorithm", pathHashAlgorithm.name());
        }
        return remoteCustomData;
    }

    public static Map<String, String> getRemoteStoreRepoName(DiscoveryNodes discoveryNodes) {
        Optional<DiscoveryNode> remoteNode = discoveryNodes.getNodes().values().stream().filter(DiscoveryNode::isRemoteStoreNode).findFirst();
        return remoteNode.map(RemoteStoreNodeAttribute::getDataRepoNames).orElseGet(HashMap::new);
    }

    public static ClusterState checkAndFinalizeRemoteStoreMigration(boolean isCompatibilityModeChanging, ClusterUpdateSettingsRequest request, ClusterState currentState, Logger logger) {
        if (isCompatibilityModeChanging && RemoteStoreUtils.isSwitchToStrictCompatibilityMode(request)) {
            return RemoteStoreUtils.finalizeMigration(currentState, logger);
        }
        return currentState;
    }

    /*
     * Enabled aggressive block sorting
     */
    public static ClusterState finalizeMigration(ClusterState incomingState, Logger logger) {
        Map<String, DiscoveryNode> discoveryNodeMap = incomingState.nodes().getNodes();
        if (discoveryNodeMap.isEmpty()) return incomingState;
        boolean remoteStoreEnabledNodePresent = discoveryNodeMap.values().stream().findFirst().get().isRemoteStoreNode();
        if (!remoteStoreEnabledNodePresent) {
            logger.debug("All nodes in the cluster are not remote nodes. Skipping.");
            return incomingState;
        }
        List<IndexMetadata> indicesWithoutRemoteStoreSettings = RemoteStoreUtils.getIndicesWithoutRemoteStoreSettings(incomingState, logger);
        if (indicesWithoutRemoteStoreSettings.isEmpty()) {
            logger.info("All indices in the cluster has remote store based index settings");
            return incomingState;
        }
        Metadata mutatedMetadata = RemoteStoreUtils.applyRemoteStoreSettings(incomingState, indicesWithoutRemoteStoreSettings, logger);
        return ClusterState.builder(incomingState).metadata(mutatedMetadata).build();
    }

    private static List<IndexMetadata> getIndicesWithoutRemoteStoreSettings(ClusterState clusterState, Logger logger) {
        Collection<IndexMetadata> allIndicesMetadata = clusterState.metadata().indices().values();
        if (!allIndicesMetadata.isEmpty()) {
            List<IndexMetadata> indicesWithoutRemoteSettings = allIndicesMetadata.stream().filter(idxMd -> !RemoteMigrationIndexMetadataUpdater.indexHasRemoteStoreSettings(idxMd.getSettings())).collect(Collectors.toList());
            logger.debug("Attempting to switch to strict mode. Count of indices without remote store settings {}", (Object)indicesWithoutRemoteSettings.size());
            return indicesWithoutRemoteSettings;
        }
        return Collections.emptyList();
    }

    private static Metadata applyRemoteStoreSettings(ClusterState clusterState, List<IndexMetadata> indicesWithoutRemoteStoreSettings, Logger logger) {
        Metadata.Builder metadataBuilder = Metadata.builder(clusterState.getMetadata());
        RoutingTable currentRoutingTable = clusterState.getRoutingTable();
        DiscoveryNodes currentDiscoveryNodes = clusterState.getNodes();
        Settings currentClusterSettings = clusterState.metadata().settings();
        for (IndexMetadata indexMetadata : indicesWithoutRemoteStoreSettings) {
            IndexMetadata.Builder indexMetadataBuilder = IndexMetadata.builder(indexMetadata);
            RemoteMigrationIndexMetadataUpdater indexMetadataUpdater = new RemoteMigrationIndexMetadataUpdater(currentDiscoveryNodes, currentRoutingTable, indexMetadata, currentClusterSettings, logger);
            indexMetadataUpdater.maybeAddRemoteIndexSettings(indexMetadataBuilder, indexMetadata.getIndex().getName());
            metadataBuilder.put(indexMetadataBuilder);
        }
        return metadataBuilder.build();
    }

    public static boolean isSwitchToStrictCompatibilityMode(ClusterUpdateSettingsRequest request) {
        Settings incomingSettings = Settings.builder().put(request.persistentSettings()).put(request.transientSettings()).build();
        return RemoteStoreNodeService.REMOTE_STORE_COMPATIBILITY_MODE_SETTING.get(incomingSettings) == RemoteStoreNodeService.CompatibilityMode.STRICT;
    }

    public static Set<String> getPinnedTimestampLockedFiles(List<String> metadataFiles, Set<Long> pinnedTimestampSet, Function<String, Long> getTimestampFunction, Function<String, Tuple<String, String>> prefixFunction, boolean ignorePinnedTimestampEnabledSetting) {
        return RemoteStoreUtils.getPinnedTimestampLockedFiles(metadataFiles, pinnedTimestampSet, new HashMap<Long, String>(), getTimestampFunction, prefixFunction, ignorePinnedTimestampEnabledSetting);
    }

    public static Set<String> getPinnedTimestampLockedFiles(List<String> metadataFiles, Set<Long> pinnedTimestampSet, Map<Long, String> metadataFilePinnedTimestampMap, Function<String, Long> getTimestampFunction, Function<String, Tuple<String, String>> prefixFunction) {
        return RemoteStoreUtils.getPinnedTimestampLockedFiles(metadataFiles, pinnedTimestampSet, metadataFilePinnedTimestampMap, getTimestampFunction, prefixFunction, false);
    }

    private static Set<String> getPinnedTimestampLockedFiles(List<String> metadataFiles, Set<Long> pinnedTimestampSet, Map<Long, String> metadataFilePinnedTimestampMap, Function<String, Long> getTimestampFunction, Function<String, Tuple<String, String>> prefixFunction, boolean ignorePinnedTimestampEnabledSetting) {
        HashSet<String> implicitLockedFiles = new HashSet<String>();
        if (!ignorePinnedTimestampEnabledSetting && !RemoteStoreSettings.isPinnedTimestampsEnabled()) {
            return implicitLockedFiles;
        }
        if (metadataFiles == null || metadataFiles.isEmpty() || pinnedTimestampSet == null) {
            return implicitLockedFiles;
        }
        metadataFilePinnedTimestampMap.keySet().retainAll(pinnedTimestampSet);
        TreeSet newPinnedTimestamps = new TreeSet(Collections.reverseOrder());
        for (Long pinnedTimestamp : pinnedTimestampSet) {
            String cachedFile = metadataFilePinnedTimestampMap.get(pinnedTimestamp);
            if (cachedFile != null) {
                assert (metadataFiles.contains(cachedFile)) : "Metadata files should contain [" + cachedFile + "]";
                implicitLockedFiles.add(cachedFile);
                continue;
            }
            newPinnedTimestamps.add(pinnedTimestamp);
        }
        if (newPinnedTimestamps.isEmpty()) {
            return implicitLockedFiles;
        }
        metadataFiles.sort(String::compareTo);
        RemoteStoreUtils.verifyNoMultipleWriters(metadataFiles, prefixFunction);
        Iterator timestampIterator = newPinnedTimestamps.iterator();
        Long currentPinnedTimestamp = (Long)timestampIterator.next();
        long prevMdTimestamp = Long.MAX_VALUE;
        for (String metadataFileName : metadataFiles) {
            long currentMdTimestamp = getTimestampFunction.apply(metadataFileName);
            if (currentMdTimestamp > prevMdTimestamp) continue;
            while (currentMdTimestamp <= currentPinnedTimestamp && prevMdTimestamp > currentPinnedTimestamp) {
                implicitLockedFiles.add(metadataFileName);
                if (prevMdTimestamp != Long.MAX_VALUE) {
                    metadataFilePinnedTimestampMap.put(currentPinnedTimestamp, metadataFileName);
                }
                if (!timestampIterator.hasNext()) {
                    return implicitLockedFiles;
                }
                currentPinnedTimestamp = (Long)timestampIterator.next();
            }
            prevMdTimestamp = currentMdTimestamp;
        }
        return implicitLockedFiles;
    }

    public static List<String> filterOutMetadataFilesBasedOnAge(List<String> metadataFiles, Function<String, Long> getTimestampFunction, long lastSuccessfulFetchOfPinnedTimestamps) {
        if (!RemoteStoreSettings.isPinnedTimestampsEnabled()) {
            return new ArrayList<String>(metadataFiles);
        }
        long maximumAllowedTimestamp = lastSuccessfulFetchOfPinnedTimestamps - 2L * RemoteStoreSettings.getPinnedTimestampsLookbackInterval().getMillis();
        ArrayList<String> metadataFilesWithMinAge = new ArrayList<String>();
        for (String metadataFileName : metadataFiles) {
            long metadataTimestamp = getTimestampFunction.apply(metadataFileName);
            if (metadataTimestamp >= maximumAllowedTimestamp) continue;
            metadataFilesWithMinAge.add(metadataFileName);
        }
        return metadataFilesWithMinAge;
    }

    public static boolean isPinnedTimestampStateStale() {
        if (!RemoteStoreSettings.isPinnedTimestampsEnabled()) {
            return false;
        }
        long lastSuccessfulFetchTimestamp = RemoteStorePinnedTimestampService.getPinnedTimestamps().v1();
        long staleBufferInMillis = RemoteStoreSettings.getPinnedTimestampsSchedulerInterval().millis() * 3L + RemoteStoreSettings.getPinnedTimestampsLookbackInterval().millis();
        return lastSuccessfulFetchTimestamp < System.currentTimeMillis() - staleBufferInMillis;
    }
}

