/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.service;

import de.virtimo.bpc.api.AbstractMaintenanceModeAcknowledgeEventHandler;
import de.virtimo.bpc.api.BpcService;
import de.virtimo.bpc.api.BpcServicesTracker;
import de.virtimo.bpc.api.EventRegistration;
import de.virtimo.bpc.api.exception.ServiceNotFoundException;
import de.virtimo.bpc.api.service.CoreBundleService;
import de.virtimo.bpc.api.service.OpenSearchService;
import de.virtimo.bpc.core.service.IndexCleanupService;
import de.virtimo.bpc.util.DateUtil;
import de.virtimo.bpc.util.StringUtil;
import de.virtimo.bpc.util.ThreadFactoryWithNamePrefix;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.opensearch.OpenSearchException;
import org.opensearch.client.RequestOptions;
import org.opensearch.client.RestHighLevelClient;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.index.reindex.BulkByScrollResponse;
import org.opensearch.index.reindex.DeleteByQueryRequest;
import org.opensearch.rest.RestStatus;
import org.osgi.framework.BundleContext;

public class IndexCleanupServiceImpl
implements IndexCleanupService,
BpcService {
    private static final Logger LOG = Logger.getLogger(IndexCleanupServiceImpl.class.getName());
    private ScheduledExecutorService executorService;
    private final BundleContext bundleContext;
    private final BpcServicesTracker<OpenSearchService> openSearchServiceTracker;
    private final BpcServicesTracker<CoreBundleService> coreBundleServiceTracker;
    private final EventRegistration eventRegistration;
    private final Map<String, DeleteTask> scheduledDeleteTasks;

    public IndexCleanupServiceImpl(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
        this.scheduledDeleteTasks = new HashMap<String, DeleteTask>();
        this.openSearchServiceTracker = new BpcServicesTracker<OpenSearchService>(bundleContext, OpenSearchService.class);
        this.coreBundleServiceTracker = new BpcServicesTracker<CoreBundleService>(bundleContext, CoreBundleService.class);
        this.executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryWithNamePrefix("bpc-core-index-cleanup"));
        this.eventRegistration = new EventRegistration(bundleContext);
        this.eventRegistration.forMaintenanceModeAcknowledgeEvents(new MaintenanceModeChangedEventHandler(bundleContext));
    }

    @Override
    public BundleContext getBundleContext() {
        return this.bundleContext;
    }

    @Override
    public void shutdownService() {
        LOG.info("shutdownService");
        this.eventRegistration.unregisterAllEventHandler();
        this.cancelAllScheduledDeleteTasks();
        if (this.executorService != null) {
            try {
                this.executorService.shutdownNow();
                this.executorService.awaitTermination(30L, TimeUnit.SECONDS);
                this.executorService = null;
            }
            catch (InterruptedException | RuntimeException ex) {
                LOG.log(Level.SEVERE, "Failed to shutdown the executor");
            }
        }
        BpcServicesTracker.stopAll(this);
    }

    @Override
    public void scheduleDeleteTask(String index, String timestampField, String deleteDocumentsOlderThan, int cleanupPeriod, TimeUnit cleanupPeriodUnit) {
        LOG.info("scheduleDeleteTask index=" + index + ", timestampField=" + timestampField + ", deleteDocumentsOlderThan=" + deleteDocumentsOlderThan + ", cleanupPeriod=" + cleanupPeriod + ", cleanupPeriodUnit=" + cleanupPeriodUnit);
        this.cancelAndRemoveScheduledDeleteTask(index);
        if (StringUtil.isNullOrEmpty(deleteDocumentsOlderThan)) {
            LOG.info("Index '" + index + "' cleanup is ignored due to missing 'deleteDocumentsOlderThan' value.");
            return;
        }
        if (cleanupPeriod <= 0) {
            LOG.info("Index '" + index + "' cleanup is ignored due to the given period of '" + cleanupPeriod + "' " + cleanupPeriodUnit + ".");
            return;
        }
        DeleteTask deleteTask = new DeleteTask(index, timestampField, deleteDocumentsOlderThan, cleanupPeriod, cleanupPeriodUnit);
        if (this.deleteTasksCanBeExecutedAtTheMoment()) {
            deleteTask.scheduleByUsingExecutorService(this.executorService);
        }
        this.scheduledDeleteTasks.put(index, deleteTask);
        LOG.info("Index '" + index + "' cleanup is done every '" + cleanupPeriod + "' " + cleanupPeriodUnit + ".");
    }

    @Override
    public void cancelAndRemoveScheduledDeleteTask(String index) {
        LOG.info("cancelAndRemoveScheduledDeleteTask index=" + index);
        if (!StringUtil.isNullOrEmpty(index) && this.scheduledDeleteTasks.containsKey(index)) {
            LOG.info("Cancelling the delete task for the index: " + index);
            DeleteTask removedDeleteTask = this.scheduledDeleteTasks.remove(index);
            removedDeleteTask.cancel();
        } else {
            LOG.info("Nothing to do, there is no delete task for the index: " + index);
        }
    }

    private boolean deleteTasksCanBeExecutedAtTheMoment() {
        try {
            if (this.coreBundleServiceTracker.getService().isMaintenanceModeEnabled()) {
                return false;
            }
        }
        catch (ServiceNotFoundException ex) {
            LOG.log(Level.SEVERE, "Failed to get the core bundle service to check the maintenance mode state.", ex);
            return false;
        }
        return true;
    }

    private void cancelAllScheduledDeleteTasks() {
        LOG.info("cancelAllScheduledDeleteTasks");
        try {
            for (DeleteTask scheduledDeleteTask : this.scheduledDeleteTasks.values()) {
                scheduledDeleteTask.cancel();
            }
        }
        catch (Exception ex) {
            LOG.log(Level.WARNING, "Failed to stop/cancel the running index cleanup tasks.", ex);
        }
    }

    private void restartAllScheduledDeleteTasks() {
        LOG.info("restartAllScheduledDeleteTasks");
        this.cancelAllScheduledDeleteTasks();
        if (this.deleteTasksCanBeExecutedAtTheMoment()) {
            for (DeleteTask scheduledDeleteTask : this.scheduledDeleteTasks.values()) {
                scheduledDeleteTask.scheduleByUsingExecutorService(this.executorService);
            }
        }
    }

    private class MaintenanceModeChangedEventHandler
    extends AbstractMaintenanceModeAcknowledgeEventHandler {
        public MaintenanceModeChangedEventHandler(BundleContext bundleContext) {
            super(bundleContext, "_core");
        }

        @Override
        public void processNewMaintenanceMode(boolean enabled) {
            LOG.info(this.getClass().getSimpleName() + ".processNewMaintenanceMode enabled=" + enabled);
            try {
                if (enabled) {
                    IndexCleanupServiceImpl.this.cancelAllScheduledDeleteTasks();
                } else {
                    IndexCleanupServiceImpl.this.restartAllScheduledDeleteTasks();
                }
            }
            catch (Exception ex) {
                LOG.log(Level.SEVERE, "Failed to process the maintenance mode changed event.", ex);
            }
        }
    }

    private class DeleteTask
    implements Runnable {
        private final String index;
        private final String timestampField;
        private final String deleteDocumentsOlderThan;
        private final int cleanupPeriod;
        private final TimeUnit cleanupPeriodUnit;
        private ScheduledFuture<?> handle;

        public DeleteTask(String index, String timestampField, String deleteDocumentsOlderThan, int cleanupPeriod, TimeUnit cleanupPeriodUnit) {
            this.index = index;
            this.timestampField = timestampField;
            this.deleteDocumentsOlderThan = deleteDocumentsOlderThan;
            this.cleanupPeriod = cleanupPeriod;
            this.cleanupPeriodUnit = cleanupPeriodUnit;
            this.handle = null;
        }

        public void scheduleByUsingExecutorService(ScheduledExecutorService executorService) {
            this.handle = executorService.scheduleWithFixedDelay(this, 60L, this.cleanupPeriod, this.cleanupPeriodUnit);
        }

        public void cancel() {
            if (this.handle != null) {
                try {
                    if (this.handle.cancel(false)) {
                        LOG.info("Running delete task cancelled. " + this);
                    }
                    LOG.warning("Failed to stop/cancel the running delete task. " + this);
                }
                finally {
                    this.handle = null;
                }
            } else {
                LOG.info("This delete task does not have a execution handle and cannot be cancelled. " + this);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            LOG.info("Cleanup of index '" + this.index + "' running ...");
            long startTimeMillis = System.currentTimeMillis();
            try {
                OpenSearchService oss = IndexCleanupServiceImpl.this.openSearchServiceTracker.getService();
                CoreBundleService coreBundleService = IndexCleanupServiceImpl.this.coreBundleServiceTracker.getService();
                RestHighLevelClient osClient = oss.getClient();
                if (!coreBundleService.isMaster()) {
                    LOG.info("Skipping the index cleanup: Must only be done by the master server.");
                    return;
                }
                if (coreBundleService.isMaintenanceModeEnabled()) {
                    LOG.info("Skipping the index cleanup: We are currently in maintenance mode.");
                    return;
                }
                Date olderThanDate = DateUtil.getDateForRelativeValue(this.deleteDocumentsOlderThan);
                DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(this.index).setQuery(QueryBuilders.rangeQuery(this.timestampField).lt(olderThanDate.getTime()));
                BulkByScrollResponse dbqrsp = osClient.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
                LOG.info("Index cleanup deleted " + dbqrsp.getDeleted() + " documents older than " + olderThanDate + " (" + olderThanDate.getTime() + ") from " + this.index);
            }
            catch (OpenSearchException ex) {
                if (ex.status() == RestStatus.NOT_FOUND) {
                    LOG.warning("No index cleanup possible, because the index '" + this.index + "' does not exist.");
                } else {
                    LOG.log(Level.SEVERE, "Unhandled exception in IndexCleanupServiceImpl.DeleteTask.run(). Please report the occurred RestStatus '" + ex.status() + "' to the BPC developers.", ex);
                }
            }
            catch (ServiceNotFoundException ex) {
                LOG.log(Level.SEVERE, "Failed to perform the " + this.index + " index cleanup, OpenSearch service is not registered.");
            }
            catch (Exception ex) {
                LOG.log(Level.SEVERE, "Failed to perform the " + this.index + " index cleanup.", ex);
            }
            finally {
                LOG.info("Index '" + this.index + "' cleanup time: " + (System.currentTimeMillis() - startTimeMillis) + "ms");
            }
        }

        public String toString() {
            return "DeleteTask{index='" + this.index + "', timestampField='" + this.timestampField + "', deleteDocumentsOlderThan='" + this.deleteDocumentsOlderThan + "', cleanupPeriod=" + this.cleanupPeriod + ", cleanupPeriodUnit=" + this.cleanupPeriodUnit + "}";
        }
    }
}

