/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.translog.transfer;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Base64;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.OutputStreamIndexOutput;
import org.opensearch.action.LatchedActionListener;
import org.opensearch.common.SetOnce;
import org.opensearch.common.blobstore.BlobMetadata;
import org.opensearch.common.blobstore.BlobPath;
import org.opensearch.common.blobstore.InputStreamWithMetadata;
import org.opensearch.common.blobstore.stream.write.WritePriority;
import org.opensearch.common.io.VersionedCodecStreamWrapper;
import org.opensearch.common.io.stream.BytesStreamOutput;
import org.opensearch.common.logging.Loggers;
import org.opensearch.common.lucene.store.ByteArrayIndexInput;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.index.shard.ShardId;
import org.opensearch.index.remote.RemoteStoreUtils;
import org.opensearch.index.remote.RemoteTranslogTransferTracker;
import org.opensearch.index.translog.Translog;
import org.opensearch.index.translog.TranslogReader;
import org.opensearch.index.translog.transfer.FileSnapshot;
import org.opensearch.index.translog.transfer.FileTransferException;
import org.opensearch.index.translog.transfer.FileTransferTracker;
import org.opensearch.index.translog.transfer.TransferService;
import org.opensearch.index.translog.transfer.TransferSnapshot;
import org.opensearch.index.translog.transfer.TranslogTransferMetadata;
import org.opensearch.index.translog.transfer.TranslogTransferMetadataHandler;
import org.opensearch.index.translog.transfer.TranslogUploadFailedException;
import org.opensearch.index.translog.transfer.listener.TranslogTransferListener;
import org.opensearch.indices.RemoteStoreSettings;

public class TranslogTransferManager {
    private final ShardId shardId;
    private final TransferService transferService;
    private final BlobPath remoteDataTransferPath;
    private final BlobPath remoteMetadataTransferPath;
    private final FileTransferTracker fileTransferTracker;
    private final RemoteTranslogTransferTracker remoteTranslogTransferTracker;
    private final RemoteStoreSettings remoteStoreSettings;
    private static final int METADATA_FILES_TO_FETCH = 10;
    private final boolean isTranslogMetadataEnabled;
    static final String CHECKPOINT_FILE_DATA_KEY = "ckp-data";
    private final Logger logger;
    private static final VersionedCodecStreamWrapper<TranslogTransferMetadata> metadataStreamWrapper = new VersionedCodecStreamWrapper<TranslogTransferMetadata>(new TranslogTransferMetadataHandler(), 1, "md");

    public TranslogTransferManager(ShardId shardId, TransferService transferService, BlobPath remoteDataTransferPath, BlobPath remoteMetadataTransferPath, FileTransferTracker fileTransferTracker, RemoteTranslogTransferTracker remoteTranslogTransferTracker, RemoteStoreSettings remoteStoreSettings, boolean isTranslogMetadataEnabled) {
        this.shardId = shardId;
        this.transferService = transferService;
        this.remoteDataTransferPath = remoteDataTransferPath;
        this.remoteMetadataTransferPath = remoteMetadataTransferPath;
        this.fileTransferTracker = fileTransferTracker;
        this.logger = Loggers.getLogger(this.getClass(), shardId, new String[0]);
        this.remoteTranslogTransferTracker = remoteTranslogTransferTracker;
        this.remoteStoreSettings = remoteStoreSettings;
        this.isTranslogMetadataEnabled = isTranslogMetadataEnabled;
    }

    public RemoteTranslogTransferTracker getRemoteTranslogTransferTracker() {
        return this.remoteTranslogTransferTracker;
    }

    public ShardId getShardId() {
        return this.shardId;
    }

    public boolean transferSnapshot(TransferSnapshot transferSnapshot, TranslogTransferListener translogTransferListener) throws IOException {
        ArrayList exceptionList = new ArrayList(transferSnapshot.getTranslogTransferMetadata().getCount());
        HashSet<FileSnapshot.TransferFileSnapshot> toUpload = new HashSet<FileSnapshot.TransferFileSnapshot>(transferSnapshot.getTranslogTransferMetadata().getCount());
        long prevUploadBytesSucceeded = this.remoteTranslogTransferTracker.getUploadBytesSucceeded();
        long prevUploadTimeInMillis = this.remoteTranslogTransferTracker.getTotalUploadTimeInMillis();
        try {
            if (this.isTranslogMetadataEnabled) {
                toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getTranslogFileSnapshotWithMetadata()));
            } else {
                toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getTranslogFileSnapshots()));
                toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getCheckpointFileSnapshots()));
            }
            if (toUpload.isEmpty()) {
                this.logger.trace("Nothing to upload for transfer");
                return true;
            }
            this.fileTransferTracker.recordBytesForFiles(toUpload);
            this.captureStatsBeforeUpload();
            CountDownLatch latch = new CountDownLatch(toUpload.size());
            LatchedActionListener<FileSnapshot.TransferFileSnapshot> latchedActionListener = new LatchedActionListener<FileSnapshot.TransferFileSnapshot>(ActionListener.wrap(this.fileTransferTracker::onSuccess, ex -> {
                assert (ex instanceof FileTransferException);
                this.logger.error(() -> new ParameterizedMessage("Exception during transfer for file {}", (Object)((FileTransferException)ex).getFileSnapshot().getName()), (Throwable)ex);
                FileTransferException e = (FileTransferException)ex;
                FileSnapshot.TransferFileSnapshot file = e.getFileSnapshot();
                this.fileTransferTracker.onFailure(file, (Exception)ex);
                exceptionList.add(ex);
            }), latch);
            HashMap<Long, BlobPath> blobPathMap = new HashMap<Long, BlobPath>();
            toUpload.forEach(fileSnapshot -> blobPathMap.put(fileSnapshot.getPrimaryTerm(), this.remoteDataTransferPath.add(String.valueOf(fileSnapshot.getPrimaryTerm()))));
            long uploadStartTime = System.nanoTime();
            this.fileTransferTracker.recordFileTransferStartTime(uploadStartTime);
            this.transferService.uploadBlobs(toUpload, blobPathMap, latchedActionListener, WritePriority.HIGH);
            try {
                if (!latch.await(this.remoteStoreSettings.getClusterRemoteTranslogTransferTimeout().millis(), TimeUnit.MILLISECONDS)) {
                    TranslogUploadFailedException ex2 = new TranslogUploadFailedException("Timed out waiting for transfer of snapshot " + String.valueOf(transferSnapshot) + " to complete");
                    exceptionList.forEach(ex2::addSuppressed);
                    throw ex2;
                }
            }
            catch (InterruptedException ex3) {
                TranslogUploadFailedException exception = new TranslogUploadFailedException("Failed to upload " + String.valueOf(transferSnapshot), ex3);
                exceptionList.forEach(exception::addSuppressed);
                Thread.currentThread().interrupt();
                throw exception;
            }
            if (exceptionList.isEmpty()) {
                FileSnapshot.TransferFileSnapshot tlogMetadata = this.prepareMetadata(transferSnapshot);
                long metadataBytesToUpload = tlogMetadata.getContentLength();
                this.remoteTranslogTransferTracker.addUploadBytesStarted(metadataBytesToUpload);
                long metadataUploadStartTime = System.nanoTime();
                try {
                    this.transferService.uploadBlob(tlogMetadata, this.remoteMetadataTransferPath, WritePriority.HIGH);
                }
                catch (Exception exception) {
                    this.remoteTranslogTransferTracker.addUploadTimeInMillis((System.nanoTime() - metadataUploadStartTime) / 1000000L);
                    this.remoteTranslogTransferTracker.addUploadBytesFailed(metadataBytesToUpload);
                    throw new TranslogUploadFailedException("Failed to upload " + tlogMetadata.getName(), exception);
                }
                this.remoteTranslogTransferTracker.addUploadTimeInMillis((System.nanoTime() - metadataUploadStartTime) / 1000000L);
                this.remoteTranslogTransferTracker.addUploadBytesSucceeded(metadataBytesToUpload);
                this.captureStatsOnUploadSuccess(prevUploadBytesSucceeded, prevUploadTimeInMillis);
                translogTransferListener.onUploadComplete(transferSnapshot);
                return true;
            }
            TranslogUploadFailedException ex4 = new TranslogUploadFailedException("Failed to upload " + exceptionList.size() + " files during transfer");
            exceptionList.forEach(ex4::addSuppressed);
            throw ex4;
        }
        catch (Exception ex5) {
            this.logger.error(() -> new ParameterizedMessage("Transfer failed for snapshot {}", (Object)transferSnapshot), (Throwable)ex5);
            this.captureStatsOnUploadFailure();
            translogTransferListener.onUploadFailed(transferSnapshot, ex5);
            return false;
        }
    }

    private void captureStatsBeforeUpload() {
        this.remoteTranslogTransferTracker.incrementTotalUploadsStarted();
        this.remoteTranslogTransferTracker.addUploadBytesStarted(this.fileTransferTracker.getTotalBytesToUpload());
    }

    private void captureStatsOnUploadSuccess(long prevUploadBytesSucceeded, long prevUploadTimeInMillis) {
        this.remoteTranslogTransferTracker.setLastSuccessfulUploadTimestamp(System.currentTimeMillis());
        this.remoteTranslogTransferTracker.incrementTotalUploadsSucceeded();
        long totalUploadedBytes = this.remoteTranslogTransferTracker.getUploadBytesSucceeded() - prevUploadBytesSucceeded;
        this.remoteTranslogTransferTracker.updateUploadBytesMovingAverage(totalUploadedBytes);
        long uploadDurationInMillis = this.remoteTranslogTransferTracker.getTotalUploadTimeInMillis() - prevUploadTimeInMillis;
        this.remoteTranslogTransferTracker.updateUploadTimeMovingAverage(uploadDurationInMillis);
        if (uploadDurationInMillis > 0L) {
            this.remoteTranslogTransferTracker.updateUploadBytesPerSecMovingAverage(totalUploadedBytes * 1000L / uploadDurationInMillis);
        }
    }

    private void captureStatsOnUploadFailure() {
        this.remoteTranslogTransferTracker.incrementTotalUploadsFailed();
    }

    public boolean downloadTranslog(String primaryTerm, String generation, Path location) throws IOException {
        this.logger.trace("Downloading translog files with: Primary Term = {}, Generation = {}, Location = {}", (Object)primaryTerm, (Object)generation, (Object)location);
        String ckpFileName = Translog.getCommitCheckpointFileName(Long.parseLong(generation));
        String translogFilename = Translog.getFilename(Long.parseLong(generation));
        if (!this.isTranslogMetadataEnabled) {
            this.downloadToFS(ckpFileName, location, primaryTerm, false);
            this.downloadToFS(translogFilename, location, primaryTerm, false);
        } else {
            Map<String, String> metadata = this.downloadToFS(translogFilename, location, primaryTerm, true);
            try {
                assert (metadata != null && !metadata.isEmpty() && metadata.containsKey(CHECKPOINT_FILE_DATA_KEY));
                this.recoverCkpFileUsingMetadata(metadata, location, generation, translogFilename);
            }
            catch (Exception e) {
                throw new IOException("Failed to recover checkpoint file from remote", e);
            }
        }
        return true;
    }

    private void recoverCkpFileUsingMetadata(Map<String, String> metadata, Path location, String generation, String fileName) throws IOException {
        String ckpFileName = Translog.getCommitCheckpointFileName(Long.parseLong(generation));
        Path filePath = location.resolve(ckpFileName);
        this.deleteFileIfExists(filePath);
        String ckpDataBase64 = metadata.get(CHECKPOINT_FILE_DATA_KEY);
        if (ckpDataBase64 == null) {
            this.logger.error("Error processing metadata for translog file: {}", (Object)fileName);
            throw new IllegalStateException("Checkpoint file data key ckp-data is expected but not found in metadata for file: " + fileName);
        }
        byte[] ckpFileBytes = Base64.getDecoder().decode(ckpDataBase64);
        Files.write(filePath, ckpFileBytes, new OpenOption[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<String, String> downloadToFS(String fileName, Path location, String primaryTerm, boolean withMetadata) throws IOException {
        Map<String, String> metadata;
        block17: {
            Path filePath = location.resolve(fileName);
            this.deleteFileIfExists(filePath);
            metadata = null;
            boolean downloadStatus = false;
            long bytesToRead = 0L;
            long downloadStartTime = System.nanoTime();
            try {
                if (withMetadata) {
                    try (InputStreamWithMetadata inputStreamWithMetadata = this.transferService.downloadBlobWithMetadata(this.remoteDataTransferPath.add(primaryTerm), fileName);){
                        InputStream inputStream = inputStreamWithMetadata.getInputStream();
                        metadata = inputStreamWithMetadata.getMetadata();
                        bytesToRead = inputStream.available();
                        Files.copy(inputStream, filePath, new CopyOption[0]);
                        downloadStatus = true;
                        break block17;
                    }
                }
                try (InputStream inputStream = this.transferService.downloadBlob(this.remoteDataTransferPath.add(primaryTerm), fileName);){
                    bytesToRead = inputStream.available();
                    Files.copy(inputStream, filePath, new CopyOption[0]);
                    downloadStatus = true;
                }
            }
            finally {
                this.remoteTranslogTransferTracker.addDownloadTimeInMillis((System.nanoTime() - downloadStartTime) / 1000000L);
                if (downloadStatus) {
                    this.remoteTranslogTransferTracker.addDownloadBytesSucceeded(bytesToRead);
                }
            }
        }
        this.fileTransferTracker.add(fileName, true);
        return metadata;
    }

    private void deleteFileIfExists(Path filePath) throws IOException {
        if (Files.exists(filePath, new LinkOption[0])) {
            Files.delete(filePath);
        }
    }

    public TranslogTransferMetadata readMetadata(long pinnedTimestamp) throws IOException {
        if (pinnedTimestamp <= 0L) {
            return this.readMetadata();
        }
        return this.readMetadata(blobMetadataList -> {
            List<String> metadataFiles = blobMetadataList.stream().map(BlobMetadata::name).collect(Collectors.toList());
            Set<String> metadataFilesMatchingTimestamp = RemoteStoreUtils.getPinnedTimestampLockedFiles(metadataFiles, Set.of(Long.valueOf(pinnedTimestamp)), file -> RemoteStoreUtils.invertLong(file.split("__")[3]), TranslogTransferMetadata::getNodeIdByPrimaryTermAndGen, true);
            if (metadataFilesMatchingTimestamp.isEmpty()) {
                return null;
            }
            assert (metadataFilesMatchingTimestamp.size() == 1) : "There should be only 1 metadata file matching given timestamp";
            return (String)metadataFilesMatchingTimestamp.stream().findFirst().get();
        }, Integer.MAX_VALUE);
    }

    public TranslogTransferMetadata readMetadata() throws IOException {
        return this.readMetadata(blobMetadataList -> {
            RemoteStoreUtils.verifyNoMultipleWriters(blobMetadataList.stream().map(BlobMetadata::name).collect(Collectors.toList()), TranslogTransferMetadata::getNodeIdByPrimaryTermAndGen);
            return ((BlobMetadata)blobMetadataList.get(0)).name();
        }, 10);
    }

    private TranslogTransferMetadata readMetadata(Function<List<BlobMetadata>, String> getMetadataFileToRead, int numberOfFilesToFetch) throws IOException {
        SetOnce metadataSetOnce = new SetOnce();
        SetOnce exceptionSetOnce = new SetOnce();
        CountDownLatch latch = new CountDownLatch(1);
        LatchedActionListener<List<BlobMetadata>> latchedActionListener = new LatchedActionListener<List<BlobMetadata>>(ActionListener.wrap(blobMetadataList -> {
            if (blobMetadataList.isEmpty()) {
                return;
            }
            String filename = (String)getMetadataFileToRead.apply((List<BlobMetadata>)blobMetadataList);
            if (filename == null) {
                return;
            }
            try {
                metadataSetOnce.set(this.readMetadata(filename));
            }
            catch (IOException e) {
                this.logger.error(() -> new ParameterizedMessage("Exception while reading metadata file: {}", (Object)filename), (Throwable)e);
                exceptionSetOnce.set(e);
            }
        }, e -> {
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            this.logger.error(() -> new ParameterizedMessage("Exception while listing metadata files", new Object[0]), (Throwable)e);
            exceptionSetOnce.set((IOException)e);
        }), latch);
        try {
            this.transferService.listAllInSortedOrder(this.remoteMetadataTransferPath, "metadata", numberOfFilesToFetch, latchedActionListener);
            if (!latch.await(this.remoteStoreSettings.getClusterRemoteTranslogTransferTimeout().millis(), TimeUnit.MILLISECONDS)) {
                throw new RuntimeException("Timed out reading metadata file");
            }
        }
        catch (InterruptedException e2) {
            throw new IOException("Exception while reading/downloading metadata file", e2);
        }
        if (exceptionSetOnce.get() != null) {
            throw (IOException)exceptionSetOnce.get();
        }
        return (TranslogTransferMetadata)metadataSetOnce.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TranslogTransferMetadata readMetadata(String metadataFilename) throws IOException {
        boolean downloadStatus = false;
        TranslogTransferMetadata translogTransferMetadata = null;
        long downloadStartTime = System.nanoTime();
        long bytesToRead = 0L;
        try (InputStream inputStream = this.transferService.downloadBlob(this.remoteMetadataTransferPath, metadataFilename);){
            bytesToRead = inputStream.available();
            ByteArrayIndexInput indexInput = new ByteArrayIndexInput("metadata file", inputStream.readAllBytes());
            translogTransferMetadata = metadataStreamWrapper.readStream(indexInput);
            downloadStatus = true;
        }
        finally {
            this.remoteTranslogTransferTracker.addDownloadTimeInMillis((System.nanoTime() - downloadStartTime) / 1000000L);
            this.logger.debug("translogMetadataDownloadStatus={}", (Object)downloadStatus);
            if (downloadStatus) {
                this.remoteTranslogTransferTracker.addDownloadBytesSucceeded(bytesToRead);
            }
        }
        return translogTransferMetadata;
    }

    private FileSnapshot.TransferFileSnapshot prepareMetadata(TransferSnapshot transferSnapshot) throws IOException {
        Map<String, String> generationPrimaryTermMap = transferSnapshot.getTranslogFileSnapshots().stream().map(s -> {
            assert (s instanceof FileSnapshot.TranslogFileSnapshot);
            return (FileSnapshot.TranslogFileSnapshot)s;
        }).collect(Collectors.toMap(snapshot -> String.valueOf(snapshot.getGeneration()), snapshot -> String.valueOf(snapshot.getPrimaryTerm())));
        TranslogTransferMetadata translogTransferMetadata = transferSnapshot.getTranslogTransferMetadata();
        translogTransferMetadata.setGenerationToPrimaryTermMapper(new HashMap<String, String>(generationPrimaryTermMap));
        return new FileSnapshot.TransferFileSnapshot(translogTransferMetadata.getFileName(), this.getMetadataBytes(translogTransferMetadata), translogTransferMetadata.getPrimaryTerm());
    }

    public byte[] getMetadataBytes(TranslogTransferMetadata metadata) throws IOException {
        byte[] metadataBytes;
        try (BytesStreamOutput output = new BytesStreamOutput();){
            try (OutputStreamIndexOutput indexOutput = new OutputStreamIndexOutput("translog transfer metadata " + metadata.getPrimaryTerm(), metadata.getFileName(), output, 4096);){
                metadataStreamWrapper.writeStream(indexOutput, metadata);
            }
            metadataBytes = BytesReference.toBytes(output.bytes());
        }
        return metadataBytes;
    }

    public void deleteGenerationAsync(long primaryTerm, Set<Long> generations, Runnable onCompletion) {
        try {
            ArrayList<String> translogFiles = new ArrayList<String>();
            generations.forEach(generation -> {
                String ckpFileName = Translog.getCommitCheckpointFileName(generation);
                String translogFileName = Translog.getFilename(generation);
                if (!this.isTranslogMetadataEnabled) {
                    translogFiles.addAll(List.of(ckpFileName, translogFileName));
                } else {
                    translogFiles.add(translogFileName);
                }
            });
            this.deleteTranslogFilesAsync(primaryTerm, translogFiles, onCompletion);
        }
        catch (Exception e) {
            onCompletion.run();
            throw e;
        }
    }

    public void deletePrimaryTermsAsync(final long minPrimaryTermToKeep) {
        this.logger.info("Deleting primary terms from remote store lesser than {}", (Object)minPrimaryTermToKeep);
        this.transferService.listFoldersAsync("remote_purge", this.remoteDataTransferPath, new ActionListener<Set<String>>(){

            @Override
            public void onResponse(Set<String> folders) {
                Set primaryTermsInRemote = folders.stream().filter(folderName -> {
                    try {
                        Long.parseLong(folderName);
                        return true;
                    }
                    catch (Exception exception) {
                        return false;
                    }
                }).map(Long::parseLong).collect(Collectors.toSet());
                Set<Long> primaryTermsToDelete = primaryTermsInRemote.stream().filter(term -> term < minPrimaryTermToKeep).collect(Collectors.toSet());
                primaryTermsToDelete.forEach(term -> TranslogTransferManager.this.deletePrimaryTermAsync((long)term));
            }

            @Override
            public void onFailure(Exception e) {
                TranslogTransferManager.this.logger.error("Exception occurred while getting primary terms from remote store", (Throwable)e);
            }
        });
    }

    public Set<Long> listPrimaryTermsInRemote() throws IOException {
        Set<String> primaryTermsStr = this.transferService.listFolders(this.remoteDataTransferPath);
        if (primaryTermsStr != null) {
            return primaryTermsStr.stream().map(Long::parseLong).collect(Collectors.toSet());
        }
        return new HashSet<Long>();
    }

    private void deletePrimaryTermAsync(final long primaryTerm) {
        this.transferService.deleteAsync("remote_purge", this.remoteDataTransferPath.add(String.valueOf(primaryTerm)), new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                TranslogTransferManager.this.logger.info("Deleted primary term {}", (Object)primaryTerm);
            }

            @Override
            public void onFailure(Exception e) {
                TranslogTransferManager.this.logger.error((Message)new ParameterizedMessage("Exception occurred while deleting primary term {}", (Object)primaryTerm), (Throwable)e);
            }
        });
    }

    public void delete() {
        this.delete(this.remoteDataTransferPath);
        this.delete(this.remoteMetadataTransferPath);
    }

    private void delete(final BlobPath path) {
        this.transferService.deleteAsync("remote_purge", path, new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                TranslogTransferManager.this.logger.info("Deleted all remote translog data at path={}", (Object)path);
            }

            @Override
            public void onFailure(Exception e) {
                TranslogTransferManager.this.logger.error((Message)new ParameterizedMessage("Exception occurred while cleaning translog at path={}", (Object)path), (Throwable)e);
            }
        });
    }

    public void listTranslogMetadataFilesAsync(ActionListener<List<BlobMetadata>> listener) {
        this.transferService.listAllInSortedOrderAsync("remote_purge", this.remoteMetadataTransferPath, "metadata", Integer.MAX_VALUE, listener);
    }

    public void deleteStaleTranslogMetadataFilesAsync(final Runnable onCompletion) {
        try {
            this.transferService.listAllInSortedOrderAsync("remote_purge", this.remoteMetadataTransferPath, "metadata", Integer.MAX_VALUE, new ActionListener<List<BlobMetadata>>(){

                @Override
                public void onResponse(List<BlobMetadata> blobMetadata) {
                    List sortedMetadataFiles = blobMetadata.stream().map(BlobMetadata::name).collect(Collectors.toList());
                    if (sortedMetadataFiles.size() <= 1) {
                        TranslogTransferManager.this.logger.trace("Remote Metadata file count is {}, so skipping deletion", (Object)sortedMetadataFiles.size());
                        onCompletion.run();
                        return;
                    }
                    List<String> metadataFilesToDelete = sortedMetadataFiles.subList(1, sortedMetadataFiles.size());
                    TranslogTransferManager.this.logger.trace("Deleting remote translog metadata files {}", metadataFilesToDelete);
                    TranslogTransferManager.this.deleteMetadataFilesAsync(metadataFilesToDelete, onCompletion);
                }

                @Override
                public void onFailure(Exception e) {
                    TranslogTransferManager.this.logger.error("Exception occurred while listing translog metadata files from remote store", (Throwable)e);
                    onCompletion.run();
                }
            });
        }
        catch (Exception e) {
            this.logger.error("Exception occurred while listing translog metadata files from remote store", (Throwable)e);
            onCompletion.run();
        }
    }

    public void deleteTranslogFiles() throws IOException {
        this.transferService.delete(this.remoteMetadataTransferPath);
        this.transferService.delete(this.remoteDataTransferPath);
    }

    private void deleteTranslogFilesAsync(final long primaryTerm, final List<String> files, final Runnable onCompletion) {
        this.transferService.deleteBlobsAsync("remote_purge", this.remoteDataTransferPath.add(String.valueOf(primaryTerm)), files, new ActionListener<Void>(){

            @Override
            public void onResponse(Void unused) {
                TranslogTransferManager.this.fileTransferTracker.delete(files);
                TranslogTransferManager.this.logger.trace("Deleted translogs for primaryTerm={} files={}", (Object)primaryTerm, (Object)files);
                onCompletion.run();
            }

            @Override
            public void onFailure(Exception e) {
                onCompletion.run();
                TranslogTransferManager.this.logger.error(() -> new ParameterizedMessage("Exception occurred while deleting translog for primaryTerm={} files={}", (Object)primaryTerm, (Object)files), (Throwable)e);
            }
        });
    }

    public void deleteMetadataFilesAsync(final List<String> files, final Runnable onCompletion) {
        try {
            this.transferService.deleteBlobsAsync("remote_purge", this.remoteMetadataTransferPath, files, new ActionListener<Void>(){

                @Override
                public void onResponse(Void unused) {
                    onCompletion.run();
                    TranslogTransferManager.this.logger.trace("Deleted remote translog metadata files {}", (Object)files);
                }

                @Override
                public void onFailure(Exception e) {
                    onCompletion.run();
                    TranslogTransferManager.this.logger.error((Message)new ParameterizedMessage("Exception occurred while deleting remote translog metadata files {}", (Object)files), (Throwable)e);
                }
            });
        }
        catch (Exception e) {
            onCompletion.run();
            throw e;
        }
    }

    public int getMaxRemoteTranslogReadersSettings() {
        return this.remoteStoreSettings.getMaxRemoteTranslogReaders();
    }

    public void populateFileTrackerWithLocalState(List<TranslogReader> readers) {
        if (readers == null) {
            return;
        }
        for (TranslogReader reader : readers) {
            long generation = reader.getGeneration();
            String tlogFilename = Translog.getFilename(generation);
            this.fileTransferTracker.add(tlogFilename, true);
            if (!this.isTranslogMetadataEnabled) continue;
            String ckpFilename = Translog.getCommitCheckpointFileName(generation);
            this.fileTransferTracker.add(ckpFilename, true);
        }
    }

    protected FileTransferTracker getFileTransferTracker() {
        return this.fileTransferTracker;
    }
}

