/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.backup;

import de.virtimo.bpc.api.ErrorCode;
import de.virtimo.bpc.api.backup.BackupSnapshotInfo;
import de.virtimo.bpc.api.backup.SnapshotName;
import de.virtimo.bpc.api.backup.exception.BackupConflictException;
import de.virtimo.bpc.api.backup.exception.BackupException;
import de.virtimo.bpc.api.exception.ModuleNotFoundException;
import de.virtimo.bpc.api.exception.OpenSearchIndexNotFoundException;
import de.virtimo.bpc.api.exception.OpenSearchRelatedException;
import de.virtimo.bpc.api.exception.ServiceNotFoundException;
import de.virtimo.bpc.api.opensearch.BpcIndexCreateCallable;
import de.virtimo.bpc.api.opensearch.BpcIndexState;
import de.virtimo.bpc.api.opensearch.BpcIndexStateException;
import de.virtimo.bpc.api.service.OpenSearchService;
import de.virtimo.bpc.core.backup.Snapshotter;
import de.virtimo.bpc.core.backup.TemporaryIndices;
import de.virtimo.bpc.core.exception.CoreErrorCode;
import de.virtimo.bpc.util.MapUtil;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.bulk.BulkRequest;
import org.opensearch.action.bulk.BulkResponse;
import org.opensearch.action.delete.DeleteRequest;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.search.SearchRequest;
import org.opensearch.client.RequestOptions;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.core.xcontent.MediaType;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.search.builder.SearchSourceBuilder;

public class SnapshotIndexModelVersionManager {
    private static final Logger LOGGER = LogManager.getLogger(SnapshotIndexModelVersionManager.class);
    private static final String ALIAS_OF_CACHE_INDEX = "bpc-snapshot-index-model-versions";
    private final OpenSearchService oss;

    public SnapshotIndexModelVersionManager(OpenSearchService oss) {
        this.oss = oss;
    }

    public void setModelVersionTo(BackupSnapshotInfo snapshotInfo) {
        LOGGER.info("setModelVersionTo snapshotInfo={}", (Object)snapshotInfo);
        if (snapshotInfo == null) {
            return;
        }
        if (!snapshotInfo.isBpcConfigurationIndex()) {
            return;
        }
        if (snapshotInfo.hasIndexModelVersion()) {
            return;
        }
        try {
            Map<SnapshotName, Integer> indexModelVersionsCache = this.readIndexModelVersionsFromCache();
            if (!indexModelVersionsCache.containsKey(snapshotInfo.getSnapshotName())) {
                Integer indexModelVersionFromRestoredSnapshot = this.restoreSnapshotAndReadIndexModelVersion(snapshotInfo);
                this.addIndexModelVersionToCache(snapshotInfo.getSnapshotName(), indexModelVersionFromRestoredSnapshot);
                indexModelVersionsCache.put(snapshotInfo.getSnapshotName(), indexModelVersionFromRestoredSnapshot);
            }
            Integer indexModelVersion = indexModelVersionsCache.get(snapshotInfo.getSnapshotName());
            snapshotInfo.setIndexModelVersion(indexModelVersion);
        }
        catch (Exception ex) {
            LOGGER.error("Error while reading index model versions from the caching index '{}'.", (Object)ALIAS_OF_CACHE_INDEX, (Object)ex);
        }
    }

    public void setModelVersionsTo(List<BackupSnapshotInfo> allExistingSnapshotInfos) {
        LOGGER.info("setModelVersionsTo allExistingSnapshotInfos=...");
        try {
            Map<SnapshotName, Integer> oldIndexModelVersionsCache = this.readIndexModelVersionsFromCache();
            HashMap<SnapshotName, Integer> newIndexModelVersionsCache = new HashMap<SnapshotName, Integer>();
            if (allExistingSnapshotInfos != null) {
                for (BackupSnapshotInfo snapshotInfo : allExistingSnapshotInfos) {
                    if (!snapshotInfo.isBpcConfigurationIndex()) continue;
                    Integer indexModelVersion = snapshotInfo.hasIndexModelVersion() ? snapshotInfo.getIndexModelVersion() : (oldIndexModelVersionsCache.containsKey(snapshotInfo.getSnapshotName()) ? oldIndexModelVersionsCache.get(snapshotInfo.getSnapshotName()) : this.restoreSnapshotAndReadIndexModelVersion(snapshotInfo));
                    if (indexModelVersion != null) {
                        newIndexModelVersionsCache.put(snapshotInfo.getSnapshotName(), indexModelVersion);
                    }
                    snapshotInfo.setIndexModelVersion(indexModelVersion);
                }
            }
            if (!oldIndexModelVersionsCache.equals(newIndexModelVersionsCache)) {
                this.writeIndexModelVersionsToCache(oldIndexModelVersionsCache, newIndexModelVersionsCache);
            }
        }
        catch (Exception ex) {
            LOGGER.error("Failed to get the index model versions of the bpc-configuration snapshots.", (Throwable)ex);
        }
    }

    public Integer addIndexModelVersionToCache(BackupSnapshotInfo snapshotInfo) {
        LOGGER.info("addIndexModelVersionToCache snapshotInfo={}", (Object)snapshotInfo);
        if (snapshotInfo == null || !snapshotInfo.isBpcConfigurationIndex()) {
            return null;
        }
        try {
            String bpcConfigurationIndexName = snapshotInfo.getIndices().get(0);
            Integer indexModelVersion = this.oss.getModelVersion(bpcConfigurationIndexName);
            this.addIndexModelVersionToCache(snapshotInfo.getSnapshotName(), indexModelVersion);
            return indexModelVersion;
        }
        catch (Exception ex) {
            LOGGER.error("Error while adding index model version from the snapshot info '{}' to the caching index '{}'.", (Object)snapshotInfo, (Object)ALIAS_OF_CACHE_INDEX, (Object)ex);
            return null;
        }
    }

    private void addIndexModelVersionToCache(SnapshotName snapshotName, Integer indexModelVersion) {
        LOGGER.info("addIndexModelVersionToCache snapshotName={}, indexModelVersion={}", (Object)snapshotName, (Object)indexModelVersion);
        if (snapshotName == null || indexModelVersion == null) {
            return;
        }
        try {
            Map<SnapshotName, Integer> oldIndexModelVersionsCache = this.readIndexModelVersionsFromCache();
            HashMap<SnapshotName, Integer> newIndexModelVersionsCache = new HashMap<SnapshotName, Integer>(oldIndexModelVersionsCache);
            newIndexModelVersionsCache.put(snapshotName, indexModelVersion);
            this.writeIndexModelVersionsToCache(oldIndexModelVersionsCache, newIndexModelVersionsCache);
        }
        catch (Exception ex) {
            LOGGER.error("Error while adding the index model version '{}' for the snapshot '{}' to the caching index '{}'.", (Object)indexModelVersion, (Object)snapshotName, (Object)ALIAS_OF_CACHE_INDEX, (Object)ex);
        }
    }

    public void deleteIndexModelVersionFromCache(SnapshotName snapshotName) {
        LOGGER.info("deleteIndexModelVersion snapshotName={}", (Object)snapshotName);
        if (snapshotName == null) {
            return;
        }
        this.deleteIndexModelVersionsFromCache(Set.of(snapshotName));
    }

    public void deleteIndexModelVersionsFromCache(Set<SnapshotName> snapshotNamesToDelete) {
        LOGGER.info("deleteIndexModelVersionsFromCache snapshotNamesToDelete={}", snapshotNamesToDelete);
        if (snapshotNamesToDelete == null) {
            return;
        }
        try {
            Map<SnapshotName, Integer> oldIndexModelVersionsCache = this.readIndexModelVersionsFromCache();
            HashMap<SnapshotName, Integer> newIndexModelVersionsCache = new HashMap<SnapshotName, Integer>();
            for (SnapshotName snapshotName : oldIndexModelVersionsCache.keySet()) {
                if (snapshotNamesToDelete.contains(snapshotName)) continue;
                newIndexModelVersionsCache.put(snapshotName, oldIndexModelVersionsCache.get(snapshotName));
            }
            this.writeIndexModelVersionsToCache(oldIndexModelVersionsCache, newIndexModelVersionsCache);
        }
        catch (Exception ex) {
            LOGGER.error("Error while deleting index model versions from the caching index '{}'.", (Object)ALIAS_OF_CACHE_INDEX, (Object)ex);
        }
    }

    private Map<SnapshotName, Integer> readIndexModelVersionsFromCache() throws OpenSearchIndexNotFoundException, OpenSearchRelatedException, BpcIndexStateException {
        LOGGER.info("readIndexModelVersionsFromCache");
        try {
            HashMap<SnapshotName, Integer> indexModelVersionsCache = new HashMap<SnapshotName, Integer>();
            BpcIndexState relatedIndexState = this.oss.getIndexState(ALIAS_OF_CACHE_INDEX);
            relatedIndexState.prepareUsing(new BpcIndexCreateCallable(){

                @Override
                public String createIndex(OpenSearchService oss) throws OpenSearchRelatedException, ServiceNotFoundException, ModuleNotFoundException {
                    return oss.getManagedIndicesHandler().createManagedIndex(SnapshotIndexModelVersionManager.ALIAS_OF_CACHE_INDEX);
                }
            });
            SearchRequest searchReq = new SearchRequest().indices(relatedIndexState.getIndexName()).source(new SearchSourceBuilder().size(500).query(QueryBuilders.matchAllQuery()));
            this.oss.search(searchReq, searchHit -> {
                Map<String, Object> sourceValues = searchHit.getSourceAsMap();
                Object snapshotNameObject = sourceValues.get("snapshot");
                Object indexModelVersionObject = sourceValues.get("indexModelVersion");
                if (snapshotNameObject instanceof String) {
                    String snapshotName = (String)snapshotNameObject;
                    if (indexModelVersionObject instanceof Integer) {
                        Integer indexModelVersion = (Integer)indexModelVersionObject;
                        indexModelVersionsCache.put(SnapshotName.parse(snapshotName), indexModelVersion);
                    }
                }
            });
            return indexModelVersionsCache;
        }
        catch (ModuleNotFoundException | ServiceNotFoundException ex) {
            throw new OpenSearchRelatedException(ex);
        }
    }

    private void writeIndexModelVersionsToCache(Map<SnapshotName, Integer> oldIndexModelVersionsCache, Map<SnapshotName, Integer> newIndexModelVersionsCache) throws OpenSearchIndexNotFoundException, OpenSearchRelatedException, BpcIndexStateException {
        LOGGER.info("writeIndexModelVersionsToCache oldIndexModelVersionsCache=..., newIndexModelVersionsCache=...");
        try {
            ArrayList<SnapshotName> deletedSnapshots = new ArrayList<SnapshotName>();
            for (SnapshotName snapshotName : oldIndexModelVersionsCache.keySet()) {
                if (newIndexModelVersionsCache.containsKey(snapshotName)) continue;
                deletedSnapshots.add(snapshotName);
            }
            HashMap<SnapshotName, Integer> indexModelVersionsToAdd = new HashMap<SnapshotName, Integer>();
            for (SnapshotName newSnapshotName : newIndexModelVersionsCache.keySet()) {
                if (oldIndexModelVersionsCache.containsKey(newSnapshotName)) continue;
                indexModelVersionsToAdd.put(newSnapshotName, newIndexModelVersionsCache.get(newSnapshotName));
            }
            if (deletedSnapshots.isEmpty() && indexModelVersionsToAdd.isEmpty()) {
                return;
            }
            BpcIndexState bpcIndexState = this.oss.getIndexState(ALIAS_OF_CACHE_INDEX);
            bpcIndexState.prepareUsing(new BpcIndexCreateCallable(){

                @Override
                public String createIndex(OpenSearchService oss) throws OpenSearchRelatedException, ServiceNotFoundException, ModuleNotFoundException {
                    return oss.getManagedIndicesHandler().createManagedIndex(SnapshotIndexModelVersionManager.ALIAS_OF_CACHE_INDEX);
                }
            });
            BulkRequest bulkRequest = new BulkRequest().timeout(TimeValue.timeValueSeconds(60L));
            for (SnapshotName snapshotName : deletedSnapshots) {
                bulkRequest.add(new DeleteRequest(ALIAS_OF_CACHE_INDEX).id(snapshotName.asString().toLowerCase()));
            }
            for (SnapshotName snapshotName : indexModelVersionsToAdd.keySet()) {
                Integer indexModelVersion = (Integer)indexModelVersionsToAdd.get(snapshotName);
                Map<String, Object> sourceValues = MapUtil.mapOf("snapshot", snapshotName.asString(), "indexModelVersion", indexModelVersion);
                bulkRequest.add(new IndexRequest(ALIAS_OF_CACHE_INDEX).id(snapshotName.asString().toLowerCase()).source(sourceValues, (MediaType)XContentType.JSON));
            }
            BulkResponse bulkResponse = this.oss.bulk(bulkRequest, RequestOptions.DEFAULT);
            if (bulkResponse.hasFailures()) {
                LOGGER.error("Updating the index model version cache index by bulk failed: {}", (Object)bulkResponse.buildFailureMessage());
            }
        }
        catch (ModuleNotFoundException | ServiceNotFoundException ex) {
            throw new OpenSearchRelatedException(ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Integer restoreSnapshotAndReadIndexModelVersion(BackupSnapshotInfo snapshotInfo) throws BackupException, BackupConflictException, OpenSearchRelatedException {
        LOGGER.info("restoreSnapshotAndReadIndexModelVersion snapshotInfo={}", (Object)snapshotInfo);
        Objects.requireNonNull(snapshotInfo, "snapshotInfo cannot be null");
        SnapshotName snapshotName = snapshotInfo.getSnapshotName();
        List<String> indices = snapshotInfo.getIndices();
        if (indices.size() != 1) {
            throw new BackupConflictException((ErrorCode)CoreErrorCode.BACKUP_RESTORE_FAILED, "Only snapshots that contain exactly one index can be processed.");
        }
        String nameOfTheOneAndOnlyIndex = indices.get(0);
        TemporaryIndices temporaryIndices = new TemporaryIndices(this.oss, indices);
        if (temporaryIndices.existsAtLeastOneOfThem()) {
            throw new BackupConflictException((ErrorCode)CoreErrorCode.BACKUP_RESTORE_FAILED, "Temporary restore not possible. There is already an index with the temporary name '${index}'.", MapUtil.mapOf("index", temporaryIndices.getNameOfExistingTmpIndex()));
        }
        Snapshotter snapshotter = new Snapshotter(this.oss);
        String nameOfTheTemporarilyRestoredIndex = temporaryIndices.getTmpIndexName(nameOfTheOneAndOnlyIndex);
        try {
            LOGGER.info("Temporary restoring the index {} of the snapshot {}.", (Object)nameOfTheTemporarilyRestoredIndex, (Object)snapshotName);
            snapshotter.restoreSnapshotIndices("bpc_backup", snapshotName, "(.+)", temporaryIndices.getPrefix() + "$1", false);
            Integer n = this.oss.getModelVersion(nameOfTheTemporarilyRestoredIndex);
            return n;
        }
        catch (OpenSearchRelatedException ex) {
            throw new BackupException((ErrorCode)CoreErrorCode.BACKUP_RESTORE_FAILED, "Failed to restore the snapshot to temporary indices.", ex);
        }
        catch (BackupException ex) {
            throw ex;
        }
        catch (Exception ex) {
            LOGGER.error("Failed to get the model version from the temporarily restored index: " + nameOfTheTemporarilyRestoredIndex, (Throwable)ex);
        }
        finally {
            temporaryIndices.deleteAll();
        }
        return null;
    }
}

