/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.indeximporter;

import de.virtimo.bpc.api.BpcServicesTracker;
import de.virtimo.bpc.api.Checker;
import de.virtimo.bpc.api.ModuleManager;
import de.virtimo.bpc.api.SystemException;
import de.virtimo.bpc.api.exception.LogServiceException;
import de.virtimo.bpc.api.exception.ModuleNotFoundException;
import de.virtimo.bpc.api.exception.OpenSearchRelatedException;
import de.virtimo.bpc.api.exception.ServiceNotFoundException;
import de.virtimo.bpc.api.opensearch.BpcIndexCreateCallable;
import de.virtimo.bpc.api.opensearch.BpcIndexState;
import de.virtimo.bpc.api.opensearch.BpcIndexStateException;
import de.virtimo.bpc.api.service.OpenSearchService;
import de.virtimo.bpc.core.CoreModule;
import de.virtimo.bpc.core.indeximporter.resource.IndexExportData;
import de.virtimo.bpc.core.utils.FelixFileinstallUtil;
import de.virtimo.bpc.util.JsonUtil;
import de.virtimo.bpc.util.ThreadFactoryWithNamePrefix;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.bulk.BulkRequest;
import org.opensearch.action.bulk.BulkResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.client.RequestOptions;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.core.xcontent.MediaType;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;

public class IndexImporter
implements Checker {
    private static final Logger LOGGER = LogManager.getLogger(IndexImporter.class);
    private static final int DEPLOYMENT_FOLDER_CHECK_RUNS_EVERY_SECONDS = 3;
    private static final String INDEX_IMPORT_FILE_EXTENSION = ".bpc-index-export";
    private static final String INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_INVALID = ".invalid_json";
    private static final String INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_ERROR = ".error";
    private static final String INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_INDEX_ALREADY_EXISTS = ".index_already_exists";
    private static final String INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_PROCESSED = ".processed";
    private static final int BULK_SIZE_LIMIT = 500;
    private ScheduledExecutorService executorService;
    private ScheduledFuture<?> checkerHandle;
    private final BundleContext bundleContext;
    private BpcServicesTracker<ModuleManager> moduleManagerTracker;
    private BpcServicesTracker<OpenSearchService> openSearchServiceTracker;
    private boolean importIsAlreadyRunning = false;

    public IndexImporter(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    @Override
    public void startChecker() {
        LOGGER.info("startChecker");
        if (this.checkerHandle != null) {
            LOGGER.warn("Index importer is already running");
            return;
        }
        BpcServicesTracker.stopAll(this);
        this.moduleManagerTracker = new BpcServicesTracker<ModuleManager>(this.bundleContext, ModuleManager.class);
        this.openSearchServiceTracker = new BpcServicesTracker<OpenSearchService>(this.bundleContext, OpenSearchService.class);
        try {
            this.executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryWithNamePrefix("bpc-core-index-importer"));
            this.checkerHandle = this.executorService.scheduleWithFixedDelay(this.createCheckerRunnable(), 0L, 3L, TimeUnit.SECONDS);
        }
        catch (Exception ex) {
            LOGGER.error("Failed to start the index importer.", (Throwable)ex);
        }
    }

    @Override
    public void stopChecker() {
        LOGGER.info("stopChecker");
        if (this.checkerHandle != null) {
            if (this.checkerHandle.cancel(true)) {
                LOGGER.info("Running index importer cancelled");
            } else {
                LOGGER.warn("Could not cancel the running index importer");
            }
            this.checkerHandle = null;
        }
        if (this.executorService != null) {
            this.executorService.shutdownNow();
            this.executorService = null;
        }
        BpcServicesTracker.stopAll(this);
    }

    private Runnable createCheckerRunnable() {
        LOGGER.info("createCheckerRunnable");
        return new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                block13: {
                    if (!IndexImporter.this.isThisServerAlreadyDefinedAsMasterOrSlave()) {
                        LOGGER.info("Skipping ... no master/slave server defined yet. A master server is necessary for the index import process.");
                        return;
                    }
                    if (IndexImporter.this.importIsAlreadyRunning) {
                        LOGGER.info("Skipping ... an index importer is already running.");
                        return;
                    }
                    IndexImporter.this.importIsAlreadyRunning = true;
                    try {
                        OpenSearchService oss;
                        File[] indexExportFiles = IndexImporter.this.getIndexExportFilesFromKarafDeployFolder();
                        if (indexExportFiles == null || indexExportFiles.length <= 0) break block13;
                        try {
                            oss = IndexImporter.this.openSearchServiceTracker.getService();
                        }
                        catch (ServiceNotFoundException ex) {
                            LOGGER.info("OpenSearch Service not ready yet. Skip index import.");
                            IndexImporter.this.importIsAlreadyRunning = false;
                            return;
                        }
                        for (File indexExportFile : indexExportFiles) {
                            try {
                                IndexImporter.this.importFile(oss, indexExportFile);
                            }
                            catch (Exception ex) {
                                LOGGER.error("Index import of file '{}' failed.", (Object)indexExportFile, (Object)ex);
                                IndexImporter.this.renameFile(indexExportFile, IndexImporter.INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_ERROR);
                            }
                        }
                    }
                    catch (Throwable t) {
                        LOGGER.error("Failed to perform the index importer.", t);
                    }
                    finally {
                        IndexImporter.this.importIsAlreadyRunning = false;
                    }
                }
            }
        };
    }

    private boolean isThisServerAlreadyDefinedAsMasterOrSlave() {
        LOGGER.info("isThisServerAlreadyDefinedAsMasterOrSlave");
        try {
            return this.moduleManagerTracker.getService().getModuleByClass(CoreModule.class).isMasterAndSlaveServerAlreadyDefined();
        }
        catch (ServiceNotFoundException e) {
            return false;
        }
        catch (ModuleNotFoundException ex) {
            return false;
        }
    }

    private File[] getIndexExportFilesFromKarafDeployFolder() throws ServiceNotFoundException, InvalidSyntaxException, IOException {
        LOGGER.info("getIndexExportFilesFromKarafDeployFolder");
        File deployFolder = FelixFileinstallUtil.getDeployFolder(this.bundleContext);
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File f, String name) {
                return name.endsWith(IndexImporter.INDEX_IMPORT_FILE_EXTENSION);
            }
        };
        return deployFolder.listFiles(filter);
    }

    private boolean renameFile(File indexExportFile, String fileExtensionPostfix) {
        LOGGER.info("renameFile indexExportFile={}, fileExtensionPostfix={}", (Object)indexExportFile, (Object)fileExtensionPostfix);
        File processedDeploymentExportFile = new File(indexExportFile.getAbsolutePath() + fileExtensionPostfix);
        return indexExportFile.renameTo(processedDeploymentExportFile);
    }

    private void importFile(OpenSearchService oss, File indexExportFile) throws SystemException {
        IndexExportData indexExportData;
        LOGGER.info("importFile indexExportFile={}", (Object)indexExportFile);
        try {
            indexExportData = JsonUtil.getInstance().convertJsonFileToPojo(indexExportFile, IndexExportData.class);
        }
        catch (Exception ex) {
            LOGGER.error("Failed to check if the file '{}' can be used for index importer. Renaming the file so that it does not get processed again.", (Object)indexExportFile, (Object)ex);
            this.renameFile(indexExportFile, INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_INVALID);
            return;
        }
        if (!indexExportData.isValid()) {
            LOGGER.error("The file '{}' is invalid. Ensure, that 'indexAlias' Renaming the file so that it does not get processed again.", (Object)indexExportFile);
            this.renameFile(indexExportFile, INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_INVALID);
            return;
        }
        if (oss.existsIndex(indexExportData.indexAlias())) {
            LOGGER.error("Index '{}' already exists. Skipping import. Renaming the file so that it does not get processed again.", (Object)indexExportData.indexAlias());
            this.renameFile(indexExportFile, INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_INDEX_ALREADY_EXISTS);
            return;
        }
        try {
            this.createIndex(oss, indexExportData);
            this.bulkInsertIndexData(oss, indexExportData);
            this.renameFile(indexExportFile, INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_PROCESSED);
        }
        catch (Exception e) {
            LOGGER.error("An error occurred while creating or filling the index '{}'. Renaming the file so that it does not get processed again.", (Object)indexExportFile, (Object)e);
            this.renameFile(indexExportFile, INDEX_IMPORT_FILE_EXTENSION_POSTFIX_WHEN_ERROR);
            try {
                this.cleanupIndexOnFailure(oss, indexExportData);
            }
            catch (OpenSearchRelatedException cleanupException) {
                LOGGER.error("Failed to delete index {} after an error occurred.", (Object)indexExportData.indexAlias(), (Object)e);
            }
        }
    }

    private void createIndex(OpenSearchService oss, final IndexExportData indexExportData) throws LogServiceException, ModuleNotFoundException, ServiceNotFoundException, OpenSearchRelatedException, BpcIndexStateException {
        LOGGER.info("createIndex oss=.., indexExportData=.. aliasName={}", (Object)indexExportData.indexAlias());
        BpcIndexState parentIndexState = oss.getIndexState(indexExportData.indexAlias());
        parentIndexState.prepareUsing(new BpcIndexCreateCallable(){

            @Override
            public String createIndex(OpenSearchService oss) throws OpenSearchRelatedException {
                return oss.createIndex(indexExportData.indexAlias(), oss.getDefaultIndexCreationSettings(), indexExportData.mappings());
            }
        });
    }

    private void bulkInsertIndexData(OpenSearchService oss, IndexExportData indexExportData) throws Exception {
        LOGGER.info("bulkInsertIndexData oss=.., indexExportData=.. aliasName={}", (Object)indexExportData.indexAlias());
        int bulkRequestSize = 0;
        BulkRequest bulkRequest = new BulkRequest().timeout(TimeValue.timeValueSeconds(60L));
        Map<String, Map<String, Object>> entries = indexExportData.data();
        for (String documentId : entries.keySet()) {
            Map<String, Object> entry = entries.get(documentId);
            IndexRequest indexRequest = ((IndexRequest)new IndexRequest().index(indexExportData.indexAlias())).id(documentId).source(entry, (MediaType)XContentType.JSON);
            bulkRequest.add(indexRequest);
            if (++bulkRequestSize != 500) continue;
            this.bulkData(oss, bulkRequest);
            bulkRequestSize = 0;
            bulkRequest = new BulkRequest().timeout(TimeValue.timeValueSeconds(60L));
        }
        if (bulkRequestSize > 0) {
            this.bulkData(oss, bulkRequest);
        }
        oss.refreshIndices(indexExportData.indexAlias());
        LOGGER.debug("OpenSearch call done");
    }

    private void bulkData(OpenSearchService oss, BulkRequest bulkRequest) throws Exception {
        BulkResponse bulkResponse = oss.bulk(bulkRequest, RequestOptions.DEFAULT);
        if (bulkResponse.hasFailures()) {
            throw new Exception("Bulk request of the given data failed. OpenSearch error: " + bulkResponse.buildFailureMessage());
        }
    }

    private void cleanupIndexOnFailure(OpenSearchService oss, IndexExportData indexExportData) throws OpenSearchRelatedException {
        LOGGER.info("cleanupIndexOnFailure oss=.., indexExportData=.. aliasName={}", (Object)indexExportData.indexAlias());
        oss.deleteIndex(indexExportData.indexAlias());
    }
}

