/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.service;

import de.virtimo.bpc.api.AbstractMaintenanceModeAcknowledgeEventHandler;
import de.virtimo.bpc.api.BpcService;
import de.virtimo.bpc.api.BpcServicesTracker;
import de.virtimo.bpc.api.EventRegistration;
import de.virtimo.bpc.api.exception.ServiceNotFoundException;
import de.virtimo.bpc.api.service.CoreBundleService;
import de.virtimo.bpc.api.service.OpenSearchService;
import de.virtimo.bpc.core.service.IndexCleanupService;
import de.virtimo.bpc.util.DateUtil;
import de.virtimo.bpc.util.StringUtil;
import de.virtimo.bpc.util.ThreadFactoryWithNamePrefix;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchException;
import org.opensearch.client.RequestOptions;
import org.opensearch.client.RestHighLevelClient;
import org.opensearch.core.rest.RestStatus;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.index.reindex.BulkByScrollResponse;
import org.opensearch.index.reindex.DeleteByQueryRequest;
import org.osgi.framework.BundleContext;

public class IndexCleanupServiceImpl
implements IndexCleanupService,
BpcService {
    private static final Logger LOGGER = LogManager.getLogger(IndexCleanupServiceImpl.class);
    private ScheduledExecutorService executorService;
    private final BundleContext bundleContext;
    private final BpcServicesTracker<OpenSearchService> openSearchServiceTracker;
    private final BpcServicesTracker<CoreBundleService> coreBundleServiceTracker;
    private final EventRegistration eventRegistration;
    private final Map<String, DeleteTask> scheduledDeleteTasks;

    public IndexCleanupServiceImpl(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
        this.scheduledDeleteTasks = new HashMap<String, DeleteTask>();
        this.openSearchServiceTracker = new BpcServicesTracker<OpenSearchService>(bundleContext, OpenSearchService.class);
        this.coreBundleServiceTracker = new BpcServicesTracker<CoreBundleService>(bundleContext, CoreBundleService.class);
        this.executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryWithNamePrefix("bpc-core-index-cleanup"));
        this.eventRegistration = new EventRegistration(bundleContext);
        this.eventRegistration.forMaintenanceModeAcknowledgeEvents(new MaintenanceModeChangedEventHandler(bundleContext));
    }

    @Override
    public BundleContext getBundleContext() {
        return this.bundleContext;
    }

    @Override
    public void shutdownService() {
        LOGGER.info("shutdownService");
        this.eventRegistration.unregisterAllEventHandler();
        this.cancelAllScheduledDeleteTasks();
        if (this.executorService != null) {
            try {
                this.executorService.shutdownNow();
                this.executorService.awaitTermination(30L, TimeUnit.SECONDS);
                this.executorService = null;
            }
            catch (InterruptedException | RuntimeException ex) {
                LOGGER.error("Failed to shutdown the executor");
            }
        }
        BpcServicesTracker.stopAll(this);
    }

    @Override
    public void scheduleDeleteTask(String index, String timestampField, String deleteDocumentsOlderThan, int cleanupPeriod, TimeUnit cleanupPeriodUnit) {
        LOGGER.info("scheduleDeleteTask index={}, timestampField={}, deleteDocumentsOlderThan={}, cleanupPeriod={}, cleanupPeriodUnit={}", (Object)index, (Object)timestampField, (Object)deleteDocumentsOlderThan, (Object)cleanupPeriod, (Object)cleanupPeriodUnit);
        this.cancelAndRemoveScheduledDeleteTask(index);
        if (StringUtil.isNullOrEmpty(deleteDocumentsOlderThan)) {
            LOGGER.info("Index '{}' cleanup is ignored due to missing 'deleteDocumentsOlderThan' value.", (Object)index);
            return;
        }
        if (cleanupPeriod <= 0) {
            LOGGER.info("Index '{}' cleanup is ignored due to the given period of {} {}.", (Object)index, (Object)cleanupPeriod, (Object)cleanupPeriodUnit);
            return;
        }
        DeleteTask deleteTask = new DeleteTask(index, timestampField, deleteDocumentsOlderThan, cleanupPeriod, cleanupPeriodUnit);
        if (this.deleteTasksCanBeExecutedAtTheMoment()) {
            deleteTask.scheduleByUsingExecutorService(this.executorService);
        }
        this.scheduledDeleteTasks.put(index, deleteTask);
        LOGGER.info("Index '{}' cleanup is done every {} {}.", (Object)index, (Object)cleanupPeriod, (Object)cleanupPeriodUnit);
    }

    @Override
    public void cancelAndRemoveScheduledDeleteTask(String index) {
        LOGGER.info("cancelAndRemoveScheduledDeleteTask index={}", (Object)index);
        if (!StringUtil.isNullOrEmpty(index) && this.scheduledDeleteTasks.containsKey(index)) {
            LOGGER.info("Cancelling the delete task for the index: {}", (Object)index);
            DeleteTask removedDeleteTask = this.scheduledDeleteTasks.remove(index);
            removedDeleteTask.cancel();
        } else {
            LOGGER.info("Nothing to do, there is no delete task for the index: {}", (Object)index);
        }
    }

    private boolean deleteTasksCanBeExecutedAtTheMoment() {
        try {
            if (this.coreBundleServiceTracker.getService().isMaintenanceModeEnabled()) {
                return false;
            }
        }
        catch (ServiceNotFoundException ex) {
            LOGGER.error("Failed to get the core bundle service to check the maintenance mode state.", (Throwable)ex);
            return false;
        }
        return true;
    }

    private void cancelAllScheduledDeleteTasks() {
        LOGGER.info("cancelAllScheduledDeleteTasks");
        try {
            for (DeleteTask scheduledDeleteTask : this.scheduledDeleteTasks.values()) {
                scheduledDeleteTask.cancel();
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Failed to stop/cancel the running index cleanup tasks.", (Throwable)ex);
        }
    }

    private void restartAllScheduledDeleteTasks() {
        LOGGER.info("restartAllScheduledDeleteTasks");
        this.cancelAllScheduledDeleteTasks();
        if (this.deleteTasksCanBeExecutedAtTheMoment()) {
            for (DeleteTask scheduledDeleteTask : this.scheduledDeleteTasks.values()) {
                scheduledDeleteTask.scheduleByUsingExecutorService(this.executorService);
            }
        }
    }

    private class MaintenanceModeChangedEventHandler
    extends AbstractMaintenanceModeAcknowledgeEventHandler {
        public MaintenanceModeChangedEventHandler(BundleContext bundleContext) {
            super(bundleContext, "_core");
        }

        @Override
        public void processNewMaintenanceMode(boolean enabled) {
            LOGGER.info("{}.processNewMaintenanceMode enabled={}", (Object)this.getClass().getSimpleName(), (Object)enabled);
            try {
                if (enabled) {
                    IndexCleanupServiceImpl.this.cancelAllScheduledDeleteTasks();
                } else {
                    IndexCleanupServiceImpl.this.restartAllScheduledDeleteTasks();
                }
            }
            catch (Exception ex) {
                LOGGER.error("Failed to process the maintenance mode changed event.", (Throwable)ex);
            }
        }
    }

    private class DeleteTask
    implements Runnable {
        private final String index;
        private final String timestampField;
        private final String deleteDocumentsOlderThan;
        private final int cleanupPeriod;
        private final TimeUnit cleanupPeriodUnit;
        private ScheduledFuture<?> handle;

        public DeleteTask(String index, String timestampField, String deleteDocumentsOlderThan, int cleanupPeriod, TimeUnit cleanupPeriodUnit) {
            this.index = index;
            this.timestampField = timestampField;
            this.deleteDocumentsOlderThan = deleteDocumentsOlderThan;
            this.cleanupPeriod = cleanupPeriod;
            this.cleanupPeriodUnit = cleanupPeriodUnit;
            this.handle = null;
        }

        public void scheduleByUsingExecutorService(ScheduledExecutorService executorService) {
            long cleanupPeriodsSeconds = this.cleanupPeriodUnit.toSeconds(this.cleanupPeriod);
            this.handle = executorService.scheduleWithFixedDelay(this, 60L, cleanupPeriodsSeconds, TimeUnit.SECONDS);
        }

        public void cancel() {
            if (this.handle != null) {
                try {
                    if (this.handle.cancel(false)) {
                        LOGGER.info("Running delete task cancelled. {}", (Object)this);
                    }
                    LOGGER.warn("Failed to stop/cancel the running delete task. {}", (Object)this);
                }
                finally {
                    this.handle = null;
                }
            } else {
                LOGGER.info("This delete task does not have a execution handle and cannot be cancelled. {}", (Object)this);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            LOGGER.info("Cleanup of index '{}' running ...", (Object)this.index);
            long startTimeMillis = System.currentTimeMillis();
            try {
                OpenSearchService oss = IndexCleanupServiceImpl.this.openSearchServiceTracker.getService();
                CoreBundleService coreBundleService = IndexCleanupServiceImpl.this.coreBundleServiceTracker.getService();
                RestHighLevelClient osClient = oss.getClient();
                if (!coreBundleService.isMaster()) {
                    LOGGER.info("Skipping the index cleanup: Must only be done by the master server.");
                    return;
                }
                if (coreBundleService.isMaintenanceModeEnabled()) {
                    LOGGER.info("Skipping the index cleanup: We are currently in maintenance mode.");
                    return;
                }
                Date olderThanDate = DateUtil.getDateForRelativeValue(this.deleteDocumentsOlderThan);
                DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(this.index).setQuery(QueryBuilders.rangeQuery(this.timestampField).lt(olderThanDate.getTime()));
                BulkByScrollResponse dbqrsp = osClient.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
                LOGGER.info("Index cleanup deleted {} documents older than {} ({}) from '{}'", (Object)dbqrsp.getDeleted(), (Object)olderThanDate, (Object)olderThanDate.getTime(), (Object)this.index);
            }
            catch (OpenSearchException ex) {
                if (ex.status() == RestStatus.NOT_FOUND) {
                    LOGGER.warn("No index cleanup possible, because the index '{}' does not exist.", (Object)this.index);
                } else {
                    LOGGER.error("Unhandled exception in IndexCleanupServiceImpl.DeleteTask.run(). Please report the occurred RestStatus '{}' to the BPC developers.", (Object)ex.status(), (Object)ex);
                }
            }
            catch (ServiceNotFoundException ex) {
                LOGGER.error("Failed to perform the '{}' index cleanup, OpenSearch service is not registered.", (Object)this.index);
            }
            catch (Exception ex) {
                LOGGER.error("Failed to perform the '{}' index cleanup.", (Object)this.index, (Object)ex);
            }
            finally {
                LOGGER.info("Index '{}' cleanup time: {} ms", (Object)this.index, (Object)(System.currentTimeMillis() - startTimeMillis));
            }
        }

        public String toString() {
            return "DeleteTask{index='" + this.index + "', timestampField='" + this.timestampField + "', deleteDocumentsOlderThan='" + this.deleteDocumentsOlderThan + "', cleanupPeriod=" + this.cleanupPeriod + ", cleanupPeriodUnit=" + this.cleanupPeriodUnit + "}";
        }
    }
}

