/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.license;

import de.virtimo.bpc.api.BpcServicesTracker;
import de.virtimo.bpc.api.Checker;
import de.virtimo.bpc.api.EventManager;
import de.virtimo.bpc.core.license.License;
import de.virtimo.bpc.core.license.LicenseService;
import de.virtimo.bpc.core.license.NoLicense;
import de.virtimo.bpc.core.notification.ExpirationNotificationEvaluator;
import de.virtimo.bpc.core.notification.Notification;
import de.virtimo.bpc.core.notification.NotificationManager;
import de.virtimo.bpc.util.SetUtil;
import de.virtimo.bpc.util.ThreadFactoryWithNamePrefix;
import java.time.LocalDate;
import java.util.ServiceConfigurationError;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.osgi.framework.BundleContext;

public class LicenseChecker
implements Checker {
    private static final Logger LOGGER = LogManager.getLogger(LicenseChecker.class);
    private ScheduledExecutorService executorService;
    private ScheduledFuture<?> checkerHandle;
    private final BundleContext bundleContext;
    private BpcServicesTracker<LicenseService> licenseServiceTracker;
    private BpcServicesTracker<EventManager> eventManagerTracker;
    private BpcServicesTracker<NotificationManager> notificationManagerTracker;

    public LicenseChecker(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    @Override
    public void startChecker() {
        LOGGER.info("startChecker");
        if (this.checkerHandle != null) {
            LOGGER.info("License checker is already running");
            return;
        }
        BpcServicesTracker.stopAll(this);
        this.licenseServiceTracker = new BpcServicesTracker<LicenseService>(this.bundleContext, LicenseService.class);
        this.eventManagerTracker = new BpcServicesTracker<EventManager>(this.bundleContext, EventManager.class);
        this.notificationManagerTracker = new BpcServicesTracker<NotificationManager>(this.bundleContext, NotificationManager.class);
        this.executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryWithNamePrefix("bpc-core-licensechecker"));
        this.checkerHandle = this.executorService.scheduleWithFixedDelay(this.createCheckerRunnable(), 30L, 60L, TimeUnit.SECONDS);
    }

    @Override
    public void stopChecker() {
        LOGGER.info("stopChecker");
        if (this.checkerHandle != null) {
            if (this.checkerHandle.cancel(true)) {
                LOGGER.info("Running license checker cancelled");
            } else {
                LOGGER.warn("Could not cancel the running license checker");
            }
            this.checkerHandle = null;
        }
        if (this.executorService != null) {
            this.executorService.shutdownNow();
            this.executorService = null;
        }
        BpcServicesTracker.stopAll(this);
    }

    private Runnable createCheckerRunnable() {
        LOGGER.info("createCheckerRunnable");
        return new Runnable(){

            @Override
            public void run() {
                LOGGER.info("LicenseChecker running ...");
                try {
                    LicenseService licenseService = LicenseChecker.this.licenseServiceTracker.getService();
                    EventManager eventManager = LicenseChecker.this.eventManagerTracker.getService();
                    NotificationManager notificationManager = LicenseChecker.this.notificationManagerTracker.getService();
                    License license = licenseService.getLicense();
                    if (license == null || license instanceof NoLicense) {
                        LOGGER.warn("Checking the NoLicense license makes no sense. Could be that the server has been started with an expired license.");
                        notificationManager.addNotification(new Notification.Builder().originator("_core").subject("License missing").message("No BPC license installed or server started with expired license").recipients(SetUtil.setOf("bpcadmin"), "role").build());
                        return;
                    }
                    if (license.isExpired()) {
                        LOGGER.warn("License expired");
                        eventManager.fireEvent("de/virtimo/bpc/core/licenseExpired", "license", license);
                        notificationManager.addNotification(new Notification.Builder().originator("_core").subject("License expired").message("Your BPC license expired").recipients(SetUtil.setOf("bpcadmin"), "role").build());
                        return;
                    }
                    LocalDate expirationDate = license.getExpirationDate();
                    if (expirationDate != null) {
                        ExpirationNotificationEvaluator expirationNotificationEvaluator = new ExpirationNotificationEvaluator(expirationDate, licenseService.getLicenseExpirationNotificationSettings());
                        if (expirationNotificationEvaluator.isTimeToNotify()) {
                            LOGGER.info("Adding license expires notification ...");
                            notificationManager.addNotification(new Notification.Builder().originator("_core").subject("License will expire in " + expirationNotificationEvaluator.getExpiresInDays() + " days").message("Your BPC license expires on " + String.valueOf(expirationDate) + " (" + expirationNotificationEvaluator.getExpiresInPeriod().getYears() + " years, " + expirationNotificationEvaluator.getExpiresInPeriod().getMonths() + " months and " + expirationNotificationEvaluator.getExpiresInPeriod().getDays() + " days)").recipients(SetUtil.setOf("bpcadmin"), "role").build());
                        } else {
                            LOGGER.info("Sending license expires notification is not necessary. Current time: {}; notifications time: {}; expires in days: {};  send only these days before: {}", (Object)expirationNotificationEvaluator.getCurrentTime(), (Object)expirationNotificationEvaluator.getSendNotificationsAtTime(), (Object)expirationNotificationEvaluator.getExpiresInPeriod().getDays(), expirationNotificationEvaluator.getSendNotificationsDaysBeforeExpiration());
                        }
                    } else {
                        LOGGER.error("Could not get the expiration date from the license. No or no valid license installed?");
                    }
                }
                catch (ServiceConfigurationError ex) {
                    LOGGER.error("Failed to check the licenses.", (Throwable)ex);
                }
                catch (Throwable t) {
                    LOGGER.error("Could not get the expiration date from the license. Unexpected exception occurred.", t);
                }
            }
        };
    }
}

