/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.misc.store;

import java.io.EOFException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.BufferedChecksum;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.FilterDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;

public class DirectIODirectory
extends FilterDirectory {
    public static final int DEFAULT_MERGE_BUFFER_SIZE = 262144;
    public static final long DEFAULT_MIN_BYTES_DIRECT = 0xA00000L;
    private final int blockSize;
    private final int mergeBufferSize;
    private final long minBytesDirect;
    volatile boolean isOpen = true;
    static final OpenOption ExtendedOpenOption_DIRECT;

    public DirectIODirectory(FSDirectory delegate, int mergeBufferSize, long minBytesDirect) throws IOException {
        super(delegate);
        this.blockSize = Math.toIntExact(Files.getFileStore(delegate.getDirectory()).getBlockSize());
        this.mergeBufferSize = mergeBufferSize;
        this.minBytesDirect = minBytesDirect;
    }

    public DirectIODirectory(FSDirectory delegate) throws IOException {
        this(delegate, 262144, 0xA00000L);
    }

    public Path getDirectory() {
        return ((FSDirectory)this.in).getDirectory();
    }

    @Override
    protected void ensureOpen() throws AlreadyClosedException {
        if (!this.isOpen) {
            throw new AlreadyClosedException("this Directory is closed");
        }
    }

    protected boolean useDirectIO(String name, IOContext context, OptionalLong fileLength) {
        return context.context() == IOContext.Context.MERGE && context.mergeInfo().estimatedMergeBytes() >= this.minBytesDirect && fileLength.orElse(this.minBytesDirect) >= this.minBytesDirect;
    }

    @Override
    public IndexInput openInput(String name, IOContext context) throws IOException {
        this.ensureOpen();
        if (this.useDirectIO(name, context, OptionalLong.of(this.fileLength(name)))) {
            return new DirectIOIndexInput(this.getDirectory().resolve(name), this.blockSize, this.mergeBufferSize);
        }
        return this.in.openInput(name, context);
    }

    @Override
    public IndexOutput createOutput(String name, IOContext context) throws IOException {
        this.ensureOpen();
        if (this.useDirectIO(name, context, OptionalLong.empty())) {
            return new DirectIOIndexOutput(this.getDirectory().resolve(name), name, this.blockSize, this.mergeBufferSize);
        }
        return this.in.createOutput(name, context);
    }

    @Override
    public void close() throws IOException {
        this.isOpen = false;
        super.close();
    }

    private static OpenOption getDirectOpenOption() {
        if (ExtendedOpenOption_DIRECT == null) {
            throw new UnsupportedOperationException("com.sun.nio.file.ExtendedOpenOption.DIRECT is not available in the current JDK version.");
        }
        return ExtendedOpenOption_DIRECT;
    }

    static {
        OpenOption option;
        try {
            Class<OpenOption> clazz = Class.forName("com.sun.nio.file.ExtendedOpenOption").asSubclass(OpenOption.class);
            option = Arrays.stream(clazz.getEnumConstants()).filter(e -> e.toString().equalsIgnoreCase("DIRECT")).findFirst().orElse(null);
        }
        catch (Exception e2) {
            option = null;
        }
        ExtendedOpenOption_DIRECT = option;
    }

    private static final class DirectIOIndexInput
    extends IndexInput {
        private final ByteBuffer buffer;
        private final FileChannel channel;
        private final int blockSize;
        private final long offset;
        private final long length;
        private final boolean isClosable;
        private boolean isOpen;
        private long filePos;

        public DirectIOIndexInput(Path path, int blockSize, int bufferSize) throws IOException {
            super("DirectIOIndexInput(path=\"" + String.valueOf(path) + "\")");
            this.channel = FileChannel.open(path, StandardOpenOption.READ, DirectIODirectory.getDirectOpenOption());
            this.blockSize = blockSize;
            this.buffer = DirectIOIndexInput.allocateBuffer(bufferSize, blockSize);
            this.isOpen = true;
            this.isClosable = true;
            this.length = this.channel.size();
            this.offset = 0L;
            this.filePos = -bufferSize;
            this.buffer.limit(0);
        }

        private DirectIOIndexInput(String description, DirectIOIndexInput other, long offset, long length) throws IOException {
            super(description);
            Objects.checkFromIndexSize(offset, length, other.channel.size());
            int bufferSize = other.buffer.capacity();
            this.buffer = DirectIOIndexInput.allocateBuffer(bufferSize, other.blockSize);
            this.blockSize = other.blockSize;
            this.channel = other.channel;
            this.isOpen = true;
            this.isClosable = false;
            this.length = length;
            this.offset = offset;
            this.filePos = -bufferSize;
            this.buffer.limit(0);
        }

        private static ByteBuffer allocateBuffer(int bufferSize, int blockSize) {
            return ByteBuffer.allocateDirect(bufferSize + blockSize - 1).alignedSlice(blockSize).order(ByteOrder.LITTLE_ENDIAN);
        }

        @Override
        public void close() throws IOException {
            if (this.isOpen && this.isClosable) {
                this.channel.close();
                this.isOpen = false;
            }
        }

        @Override
        public long getFilePointer() {
            long filePointer = this.filePos + (long)this.buffer.position() - this.offset;
            assert (filePointer == (long)(-this.buffer.capacity()) - this.offset || filePointer >= 0L) : "filePointer should either be initial value equal to negative buffer capacity, or larger than or equal to 0";
            return Math.max(filePointer, 0L);
        }

        @Override
        public void seek(long pos) throws IOException {
            if (pos != this.getFilePointer()) {
                long absolutePos = pos + this.offset;
                if (absolutePos >= this.filePos && absolutePos <= this.filePos + (long)this.buffer.limit()) {
                    this.buffer.position(Math.toIntExact(absolutePos - this.filePos));
                } else {
                    this.seekInternal(pos);
                }
            }
            assert (pos == this.getFilePointer());
        }

        private void seekInternal(long pos) throws IOException {
            long absPos = pos + this.offset;
            long alignedPos = absPos - absPos % (long)this.blockSize;
            this.filePos = alignedPos - (long)this.buffer.capacity();
            int delta = (int)(absPos - alignedPos);
            this.refill(delta);
            this.buffer.position(delta);
        }

        @Override
        public long length() {
            return this.length;
        }

        @Override
        public byte readByte() throws IOException {
            if (!this.buffer.hasRemaining()) {
                this.refill(1);
            }
            return this.buffer.get();
        }

        @Override
        public short readShort() throws IOException {
            if (this.buffer.remaining() >= 2) {
                return this.buffer.getShort();
            }
            return super.readShort();
        }

        @Override
        public int readInt() throws IOException {
            if (this.buffer.remaining() >= 4) {
                return this.buffer.getInt();
            }
            return super.readInt();
        }

        @Override
        public long readLong() throws IOException {
            if (this.buffer.remaining() >= 8) {
                return this.buffer.getLong();
            }
            return super.readLong();
        }

        private void refill(int bytesToRead) throws IOException {
            this.filePos += (long)this.buffer.capacity();
            if (this.filePos > this.offset + this.length || this.offset + this.length - this.filePos < (long)bytesToRead) {
                throw new EOFException("read past EOF: " + String.valueOf(this));
            }
            this.buffer.clear();
            try {
                this.channel.read(this.buffer, this.filePos);
            }
            catch (IOException ioe) {
                throw new IOException(ioe.getMessage() + ": " + String.valueOf(this), ioe);
            }
            this.buffer.flip();
        }

        @Override
        public void readBytes(byte[] dst, int offset, int len) throws IOException {
            int left;
            int toRead = len;
            while ((left = this.buffer.remaining()) < toRead) {
                this.buffer.get(dst, offset, left);
                offset += left;
                this.refill(toRead -= left);
            }
            this.buffer.get(dst, offset, toRead);
        }

        @Override
        public void readInts(int[] dst, int offset, int len) throws IOException {
            int remainingDst = len;
            while (remainingDst > 0) {
                int cnt = Math.min(this.buffer.remaining() / 4, remainingDst);
                this.buffer.asIntBuffer().get(dst, offset + len - remainingDst, cnt);
                this.buffer.position(this.buffer.position() + 4 * cnt);
                if ((remainingDst -= cnt) <= 0) continue;
                if (this.buffer.hasRemaining()) {
                    dst[offset + len - remainingDst] = this.readInt();
                    --remainingDst;
                    continue;
                }
                this.refill(remainingDst * 4);
            }
        }

        @Override
        public void readFloats(float[] dst, int offset, int len) throws IOException {
            int remainingDst = len;
            while (remainingDst > 0) {
                int cnt = Math.min(this.buffer.remaining() / 4, remainingDst);
                this.buffer.asFloatBuffer().get(dst, offset + len - remainingDst, cnt);
                this.buffer.position(this.buffer.position() + 4 * cnt);
                if ((remainingDst -= cnt) <= 0) continue;
                if (this.buffer.hasRemaining()) {
                    dst[offset + len - remainingDst] = Float.intBitsToFloat(this.readInt());
                    --remainingDst;
                    continue;
                }
                this.refill(remainingDst * 4);
            }
        }

        @Override
        public void readLongs(long[] dst, int offset, int len) throws IOException {
            int remainingDst = len;
            while (remainingDst > 0) {
                int cnt = Math.min(this.buffer.remaining() / 8, remainingDst);
                this.buffer.asLongBuffer().get(dst, offset + len - remainingDst, cnt);
                this.buffer.position(this.buffer.position() + 8 * cnt);
                if ((remainingDst -= cnt) <= 0) continue;
                if (this.buffer.hasRemaining()) {
                    dst[offset + len - remainingDst] = this.readLong();
                    --remainingDst;
                    continue;
                }
                this.refill(remainingDst * 8);
            }
        }

        @Override
        public DirectIOIndexInput clone() {
            try {
                DirectIOIndexInput clone = new DirectIOIndexInput("clone:" + String.valueOf(this), this, this.offset, this.length);
                clone.seekInternal(this.getFilePointer());
                return clone;
            }
            catch (IOException ioe) {
                throw new UncheckedIOException(ioe);
            }
        }

        @Override
        public IndexInput slice(String sliceDescription, long offset, long length) throws IOException {
            if ((length | offset) < 0L || length > this.length - offset) {
                throw new IllegalArgumentException("slice() " + sliceDescription + " out of bounds: " + String.valueOf(this));
            }
            DirectIOIndexInput slice = new DirectIOIndexInput(sliceDescription, this, this.offset + offset, length);
            slice.seekInternal(0L);
            return slice;
        }
    }

    private static final class DirectIOIndexOutput
    extends IndexOutput {
        private final ByteBuffer buffer;
        private final FileChannel channel;
        private final Checksum digest;
        private long filePos;
        private boolean isOpen;

        public DirectIOIndexOutput(Path path, String name, int blockSize, int bufferSize) throws IOException {
            super("DirectIOIndexOutput(path=\"" + path.toString() + "\")", name);
            this.channel = FileChannel.open(path, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW, DirectIODirectory.getDirectOpenOption());
            this.buffer = ByteBuffer.allocateDirect(bufferSize + blockSize - 1).alignedSlice(blockSize);
            this.digest = new BufferedChecksum(new CRC32());
            this.isOpen = true;
        }

        @Override
        public void writeByte(byte b) throws IOException {
            this.buffer.put(b);
            this.digest.update(b);
            if (!this.buffer.hasRemaining()) {
                this.dump();
            }
        }

        @Override
        public void writeBytes(byte[] src, int offset, int len) throws IOException {
            int left;
            int toWrite = len;
            while ((left = this.buffer.remaining()) <= toWrite) {
                this.buffer.put(src, offset, left);
                this.digest.update(src, offset, left);
                toWrite -= left;
                offset += left;
                this.dump();
            }
            this.buffer.put(src, offset, toWrite);
            this.digest.update(src, offset, toWrite);
        }

        private void dump() throws IOException {
            int size = this.buffer.position();
            this.buffer.rewind();
            this.channel.write(this.buffer, this.filePos);
            this.filePos += (long)size;
            this.buffer.clear();
        }

        @Override
        public long getFilePointer() {
            return this.filePos + (long)this.buffer.position();
        }

        @Override
        public long getChecksum() {
            return this.digest.getValue();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            if (this.isOpen) {
                this.isOpen = false;
                try {
                    this.dump();
                }
                finally {
                    try (FileChannel ch = this.channel;){
                        ch.truncate(this.getFilePointer());
                    }
                }
            }
        }
    }
}

