/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.indices.pollingingest;

import java.io.Closeable;
import java.io.IOException;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.index.Term;
import org.opensearch.action.DocWriteRequest;
import org.opensearch.common.Nullable;
import org.opensearch.common.metrics.CounterMetric;
import org.opensearch.common.xcontent.XContentFactory;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.xcontent.MediaTypeRegistry;
import org.opensearch.index.IngestionShardPointer;
import org.opensearch.index.Message;
import org.opensearch.index.VersionType;
import org.opensearch.index.engine.Engine;
import org.opensearch.index.engine.IngestionEngine;
import org.opensearch.index.engine.VersionConflictEngineException;
import org.opensearch.index.mapper.ParseContext;
import org.opensearch.index.mapper.ParsedDocument;
import org.opensearch.index.mapper.SourceToParse;
import org.opensearch.index.mapper.Uid;
import org.opensearch.indices.pollingingest.IngestionErrorStrategy;
import org.opensearch.indices.pollingingest.ShardUpdateMessage;

public class MessageProcessorRunnable
implements Runnable,
Closeable {
    private static final Logger logger = LogManager.getLogger(MessageProcessorRunnable.class);
    private static final String ID = "_id";
    private static final String OP_TYPE = "_op_type";
    private static final String SOURCE = "_source";
    private static final int MIN_RETRY_COUNT = 2;
    private static final int WAIT_BEFORE_RETRY_DURATION_MS = 2000;
    private final BlockingQueue<ShardUpdateMessage<? extends IngestionShardPointer, ? extends Message>> blockingQueue;
    private final MessageProcessor messageProcessor;
    private final MessageProcessorMetrics messageProcessorMetrics = MessageProcessorMetrics.create();
    @Nullable
    private volatile IngestionShardPointer currentShardPointer;
    private volatile boolean closed = false;
    private volatile IngestionErrorStrategy errorStrategy;

    public MessageProcessorRunnable(BlockingQueue<ShardUpdateMessage<? extends IngestionShardPointer, ? extends Message>> blockingQueue, IngestionEngine engine, IngestionErrorStrategy errorStrategy) {
        this(blockingQueue, new MessageProcessor(engine), errorStrategy);
    }

    MessageProcessorRunnable(BlockingQueue<ShardUpdateMessage<? extends IngestionShardPointer, ? extends Message>> blockingQueue, MessageProcessor messageProcessor, IngestionErrorStrategy errorStrategy) {
        this.blockingQueue = Objects.requireNonNull(blockingQueue);
        this.messageProcessor = messageProcessor;
        this.errorStrategy = errorStrategy;
    }

    private static BytesReference convertToBytes(Object object) throws IOException {
        assert (object instanceof Map);
        return BytesReference.bytes(XContentFactory.jsonBuilder().map((Map)object));
    }

    BlockingQueue<ShardUpdateMessage<? extends IngestionShardPointer, ? extends Message>> getBlockingQueue() {
        return this.blockingQueue;
    }

    @Override
    public void run() {
        ShardUpdateMessage<? extends IngestionShardPointer, ? extends Message> shardUpdateMessage = null;
        int retryCount = 0;
        while (!Thread.currentThread().isInterrupted() && !this.closed) {
            try {
                if (shardUpdateMessage == null) {
                    shardUpdateMessage = this.blockingQueue.poll(1000L, TimeUnit.MILLISECONDS);
                }
            }
            catch (InterruptedException e) {
                this.messageProcessorMetrics.processorThreadInterruptCounter.inc();
                logger.debug("MessageProcessorRunnable poll interruptedException", (Throwable)e);
                Thread.currentThread().interrupt();
            }
            if (shardUpdateMessage == null) continue;
            try {
                this.messageProcessorMetrics.processedCounter.inc();
                this.currentShardPointer = shardUpdateMessage.pointer();
                this.messageProcessor.process(shardUpdateMessage, this.messageProcessorMetrics);
                shardUpdateMessage = null;
                retryCount = 0;
            }
            catch (VersionConflictEngineException e) {
                this.messageProcessorMetrics.versionConflictCounter.inc();
                logger.debug("Dropping message due to version conflict. ShardPointer: " + shardUpdateMessage.pointer().asString(), (Throwable)e);
                shardUpdateMessage = null;
            }
            catch (Exception e) {
                boolean retriesExhausted;
                this.messageProcessorMetrics.failedMessageCounter.inc();
                this.errorStrategy.handleError(e, IngestionErrorStrategy.ErrorStage.PROCESSING);
                boolean bl = retriesExhausted = retryCount >= 2 || e instanceof IllegalArgumentException;
                if (retriesExhausted && this.errorStrategy.shouldIgnoreError(e, IngestionErrorStrategy.ErrorStage.PROCESSING)) {
                    this.logDroppedMessage(shardUpdateMessage);
                    shardUpdateMessage = null;
                    retryCount = 0;
                    this.messageProcessorMetrics.failedMessageDroppedCounter.inc();
                    continue;
                }
                ++retryCount;
                this.waitBeforeRetry();
            }
        }
    }

    private void waitBeforeRetry() {
        try {
            Thread.sleep(2000L);
        }
        catch (InterruptedException e) {
            logger.debug("MessageProcessor thread interrupted while waiting for retry", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    public MessageProcessorMetrics getMessageProcessorMetrics() {
        return this.messageProcessorMetrics;
    }

    public IngestionErrorStrategy getErrorStrategy() {
        return this.errorStrategy;
    }

    public void setErrorStrategy(IngestionErrorStrategy errorStrategy) {
        this.errorStrategy = errorStrategy;
    }

    @Nullable
    public IngestionShardPointer getCurrentShardPointer() {
        return this.currentShardPointer;
    }

    @Override
    public void close() {
        this.closed = true;
    }

    private void logDroppedMessage(ShardUpdateMessage shardUpdateMessage) {
        String id = shardUpdateMessage.autoGeneratedIdTimestamp() == -1L ? (String)shardUpdateMessage.parsedPayloadMap().get(ID) : "null";
        logger.warn("Exhausted retries, dropping message: _id:{}, pointer:{}", (Object)id, (Object)shardUpdateMessage.pointer().asString());
    }

    static class MessageProcessor {
        private final IngestionEngine engine;
        private final String index;

        MessageProcessor(IngestionEngine engine) {
            this(engine, engine.config().getIndexSettings().getIndex().getName());
        }

        MessageProcessor(IngestionEngine engine, String index) {
            this.engine = engine;
            this.index = index;
        }

        protected void process(ShardUpdateMessage shardUpdateMessage, MessageProcessorMetrics messageProcessorMetrics) {
            try {
                MessageOperation operation = this.getOperation(shardUpdateMessage, messageProcessorMetrics);
                switch (operation.engineOperation.operationType()) {
                    case INDEX: {
                        boolean isCreateMode = operation.opType == DocWriteRequest.OpType.CREATE;
                        this.engine.indexInternal((Engine.Index)operation.engineOperation, isCreateMode);
                        break;
                    }
                    case DELETE: {
                        this.engine.deleteInternal((Engine.Delete)operation.engineOperation);
                        break;
                    }
                    case NO_OP: {
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Invalid operation: " + String.valueOf(operation.engineOperation));
                    }
                }
            }
            catch (IOException e) {
                logger.error("Failed to process operation from message {} at pointer {}: {}", shardUpdateMessage.originalMessage(), (Object)shardUpdateMessage.pointer().asString(), (Object)e);
                throw new RuntimeException(e);
            }
        }

        protected MessageOperation getOperation(ShardUpdateMessage shardUpdateMessage, MessageProcessorMetrics messageProcessorMetrics) throws IOException {
            Map<String, Object> payloadMap = shardUpdateMessage.parsedPayloadMap();
            Object pointer = shardUpdateMessage.pointer();
            if (payloadMap.containsKey(MessageProcessorRunnable.OP_TYPE) && !(payloadMap.get(MessageProcessorRunnable.OP_TYPE) instanceof String)) {
                messageProcessorMetrics.invalidMessageCounter.inc();
                String errorMessage = String.format(Locale.getDefault(), "_op_type field is of type %s but not string. Invalid message.", payloadMap.get(MessageProcessorRunnable.OP_TYPE).getClass());
                logger.error(errorMessage);
                throw new IllegalArgumentException(errorMessage);
            }
            if (!payloadMap.containsKey(MessageProcessorRunnable.ID)) {
                messageProcessorMetrics.invalidMessageCounter.inc();
                String errorMessage = "ID field is missing. Invalid message.";
                logger.error(errorMessage);
                throw new IllegalArgumentException(errorMessage);
            }
            String id = (String)payloadMap.get(MessageProcessorRunnable.ID);
            String opTypeString = (String)payloadMap.getOrDefault(MessageProcessorRunnable.OP_TYPE, "index");
            DocWriteRequest.OpType opType = DocWriteRequest.OpType.fromString(opTypeString);
            long documentVersion = -3L;
            VersionType documentVersionType = VersionType.INTERNAL;
            if (payloadMap.containsKey("_version")) {
                documentVersion = Long.parseLong((String)payloadMap.get("_version"));
                documentVersionType = VersionType.EXTERNAL;
            }
            return new MessageOperation(switch (opType) {
                case DocWriteRequest.OpType.INDEX, DocWriteRequest.OpType.CREATE -> {
                    if (!payloadMap.containsKey(MessageProcessorRunnable.SOURCE)) {
                        messageProcessorMetrics.invalidMessageCounter.inc();
                        String errorMessage = "Missing _source field. Invalid message";
                        logger.error(errorMessage);
                        throw new IllegalArgumentException(errorMessage);
                    }
                    if (!(payloadMap.get(MessageProcessorRunnable.SOURCE) instanceof Map)) {
                        messageProcessorMetrics.invalidMessageCounter.inc();
                        String errorMessage = "_source field does not contain a map. Invalid message";
                        logger.error(errorMessage);
                        throw new IllegalArgumentException(errorMessage);
                    }
                    BytesReference source = MessageProcessorRunnable.convertToBytes(payloadMap.get(MessageProcessorRunnable.SOURCE));
                    SourceToParse sourceToParse = new SourceToParse(this.index, id, source, MediaTypeRegistry.xContentType(source), null);
                    ParsedDocument doc = this.engine.getDocumentMapperForType().getDocumentMapper().parse(sourceToParse);
                    ParseContext.Document document = doc.rootDoc();
                    document.add(pointer.asPointField("_offset"));
                    document.add(new StoredField("_offset", pointer.asString()));
                    yield new Engine.Index(new Term(MessageProcessorRunnable.ID, Uid.encodeId(id)), doc, 0L, 1L, documentVersion, documentVersionType, Engine.Operation.Origin.PRIMARY, System.nanoTime(), shardUpdateMessage.autoGeneratedIdTimestamp(), false, -2L, 0L);
                }
                case DocWriteRequest.OpType.DELETE -> {
                    if (shardUpdateMessage.autoGeneratedIdTimestamp() != -1L) {
                        logger.info("Delete operation without ID received, and will be dropped.");
                        messageProcessorMetrics.invalidMessageCounter.inc();
                        yield new Engine.NoOp(0L, 1L, Engine.Operation.Origin.PRIMARY, System.nanoTime(), "Delete operation is missing ID. Skipping message.");
                    }
                    yield new Engine.Delete(id, new Term(MessageProcessorRunnable.ID, Uid.encodeId(id)), 0L, 1L, documentVersion, documentVersionType, Engine.Operation.Origin.PRIMARY, System.nanoTime(), -2L, 0L);
                }
                default -> {
                    messageProcessorMetrics.invalidMessageCounter.inc();
                    logger.error("Unsupported operation type {}", (Object)opType);
                    throw new IllegalArgumentException("Unsupported operation");
                }
            }, opType);
        }
    }

    public record MessageProcessorMetrics(CounterMetric processedCounter, CounterMetric invalidMessageCounter, CounterMetric versionConflictCounter, CounterMetric failedMessageCounter, CounterMetric failedMessageDroppedCounter, CounterMetric processorThreadInterruptCounter) {
        public static MessageProcessorMetrics create() {
            return new MessageProcessorMetrics(new CounterMetric(), new CounterMetric(), new CounterMetric(), new CounterMetric(), new CounterMetric(), new CounterMetric());
        }

        public MessageProcessorMetrics combine(MessageProcessorMetrics other) {
            MessageProcessorMetrics combinedMetrics = MessageProcessorMetrics.create();
            combinedMetrics.processedCounter.inc(this.processedCounter.count() + other.processedCounter.count());
            combinedMetrics.invalidMessageCounter.inc(this.invalidMessageCounter.count() + other.invalidMessageCounter.count());
            combinedMetrics.versionConflictCounter.inc(this.versionConflictCounter.count() + other.versionConflictCounter.count());
            combinedMetrics.failedMessageCounter.inc(this.failedMessageCounter.count() + other.failedMessageCounter.count());
            combinedMetrics.failedMessageDroppedCounter.inc(this.failedMessageDroppedCounter.count() + other.failedMessageDroppedCounter.count());
            combinedMetrics.processorThreadInterruptCounter.inc(this.processorThreadInterruptCounter.count() + other.processorThreadInterruptCounter.count());
            return combinedMetrics;
        }
    }

    protected record MessageOperation(Engine.Operation engineOperation, DocWriteRequest.OpType opType) {
    }
}

