/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.translog;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.LongConsumer;
import java.util.function.LongSupplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.AlreadyClosedException;
import org.opensearch.Version;
import org.opensearch.common.Nullable;
import org.opensearch.common.UUIDs;
import org.opensearch.common.annotation.PublicApi;
import org.opensearch.common.io.stream.ReleasableBytesStreamOutput;
import org.opensearch.common.lease.Releasable;
import org.opensearch.common.lease.Releasables;
import org.opensearch.common.util.BigArrays;
import org.opensearch.common.util.concurrent.ReleasableLock;
import org.opensearch.common.util.io.IOUtils;
import org.opensearch.core.common.Strings;
import org.opensearch.core.common.bytes.BytesArray;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.common.io.stream.BufferedChecksumStreamInput;
import org.opensearch.core.common.io.stream.BufferedChecksumStreamOutput;
import org.opensearch.core.common.io.stream.StreamInput;
import org.opensearch.core.common.io.stream.StreamOutput;
import org.opensearch.core.index.shard.ShardId;
import org.opensearch.index.VersionType;
import org.opensearch.index.engine.Engine;
import org.opensearch.index.engine.MissingHistoryOperationsException;
import org.opensearch.index.mapper.Uid;
import org.opensearch.index.shard.AbstractIndexShardComponent;
import org.opensearch.index.shard.IndexShardComponent;
import org.opensearch.index.translog.BaseTranslogReader;
import org.opensearch.index.translog.ChannelFactory;
import org.opensearch.index.translog.Checkpoint;
import org.opensearch.index.translog.MultiSnapshot;
import org.opensearch.index.translog.TragicExceptionHolder;
import org.opensearch.index.translog.TranslogConfig;
import org.opensearch.index.translog.TranslogCorruptedException;
import org.opensearch.index.translog.TranslogDeletionPolicy;
import org.opensearch.index.translog.TranslogException;
import org.opensearch.index.translog.TranslogHeader;
import org.opensearch.index.translog.TranslogOperationHelper;
import org.opensearch.index.translog.TranslogReader;
import org.opensearch.index.translog.TranslogSnapshot;
import org.opensearch.index.translog.TranslogStats;
import org.opensearch.index.translog.TranslogWriter;
import org.opensearch.index.translog.TruncatedTranslogException;

@PublicApi(since="1.0.0")
public abstract class Translog
extends AbstractIndexShardComponent
implements IndexShardComponent,
Closeable {
    public static final String TRANSLOG_UUID_KEY = "translog_uuid";
    public static final String TRANSLOG_FILE_PREFIX = "translog-";
    public static final String TRANSLOG_FILE_SUFFIX = ".tlog";
    public static final String CHECKPOINT_SUFFIX = ".ckp";
    public static final String CHECKPOINT_FILE_NAME = "translog.ckp";
    static final Pattern PARSE_STRICT_ID_PATTERN = Pattern.compile("^translog-(\\d+)(\\.tlog)$");
    public static final int DEFAULT_HEADER_SIZE_IN_BYTES = TranslogHeader.headerSizeInBytes(UUIDs.randomBase64UUID());
    protected final List<TranslogReader> readers = new ArrayList<TranslogReader>();
    protected final BigArrays bigArrays;
    protected final ReleasableLock readLock;
    protected final ReleasableLock writeLock;
    protected final Path location;
    protected TranslogWriter current;
    protected final TragicExceptionHolder tragedy = new TragicExceptionHolder();
    protected final AtomicBoolean closed = new AtomicBoolean();
    protected final TranslogConfig config;
    protected final LongSupplier globalCheckpointSupplier;
    protected final LongSupplier primaryTermSupplier;
    protected final String translogUUID;
    protected final TranslogDeletionPolicy deletionPolicy;
    protected final LongConsumer persistedSequenceNumberConsumer;
    protected final TranslogOperationHelper translogOperationHelper;
    public static final Location EMPTY_TRANSLOG_LOCATION = new Location(0L, 0L, 0);
    public static final Snapshot EMPTY_TRANSLOG_SNAPSHOT = new Snapshot(){

        @Override
        public void close() {
        }

        @Override
        public int totalOperations() {
            return 0;
        }

        @Override
        public Operation next() {
            return null;
        }
    };

    public Translog(TranslogConfig config, String translogUUID, TranslogDeletionPolicy deletionPolicy, LongSupplier globalCheckpointSupplier, LongSupplier primaryTermSupplier, LongConsumer persistedSequenceNumberConsumer, TranslogOperationHelper translogOperationHelper) throws IOException {
        super(config.getShardId(), config.getIndexSettings());
        this.config = config;
        this.globalCheckpointSupplier = globalCheckpointSupplier;
        this.primaryTermSupplier = primaryTermSupplier;
        this.persistedSequenceNumberConsumer = persistedSequenceNumberConsumer;
        this.deletionPolicy = deletionPolicy;
        this.translogUUID = translogUUID;
        this.bigArrays = config.getBigArrays();
        ReentrantReadWriteLock rwl = new ReentrantReadWriteLock();
        this.readLock = new ReleasableLock(rwl.readLock());
        this.writeLock = new ReleasableLock(rwl.writeLock());
        this.location = config.getTranslogPath();
        Files.createDirectories(this.location, new FileAttribute[0]);
        this.translogOperationHelper = translogOperationHelper;
    }

    public Translog(TranslogConfig config, String translogUUID, TranslogDeletionPolicy deletionPolicy, LongSupplier globalCheckpointSupplier, LongSupplier primaryTermSupplier, LongConsumer persistedSequenceNumberConsumer) throws IOException {
        this(config, translogUUID, deletionPolicy, globalCheckpointSupplier, primaryTermSupplier, persistedSequenceNumberConsumer, TranslogOperationHelper.DEFAULT);
        assert (!config.getIndexSettings().isDerivedSourceEnabled());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected ArrayList<TranslogReader> recoverFromFiles(Checkpoint checkpoint) throws IOException {
        boolean success = false;
        ArrayList<TranslogReader> foundTranslogs = new ArrayList<TranslogReader>();
        try (ReleasableLock ignored = this.writeLock.acquire();){
            this.logger.debug("open uncommitted translog checkpoint {}", (Object)checkpoint);
            long minGenerationToRecoverFrom = checkpoint.minTranslogGeneration;
            for (long i = checkpoint.generation; i >= minGenerationToRecoverFrom; --i) {
                Path committedTranslogFile = this.location.resolve(Translog.getFilename(i));
                if (!Files.exists(committedTranslogFile, new LinkOption[0])) {
                    throw new TranslogCorruptedException(committedTranslogFile.toString(), "translog file doesn't exist with generation: " + i + " recovering from: " + minGenerationToRecoverFrom + " checkpoint: " + checkpoint.generation + " - translog ids must be consecutive");
                }
                Checkpoint readerCheckpoint = i == checkpoint.generation ? checkpoint : Checkpoint.read(this.location.resolve(Translog.getCommitCheckpointFileName(i)));
                TranslogReader reader = this.openReader(committedTranslogFile, readerCheckpoint);
                assert (reader.getPrimaryTerm() <= this.primaryTermSupplier.getAsLong()) : "Primary terms go backwards; current term [" + this.primaryTermSupplier.getAsLong() + "] translog path [ " + String.valueOf(committedTranslogFile) + ", existing term [" + reader.getPrimaryTerm() + "]";
                foundTranslogs.add(reader);
                this.logger.debug("recovered local translog from checkpoint {}", (Object)checkpoint);
            }
            Collections.reverse(foundTranslogs);
            IOUtils.deleteFilesIgnoringExceptions((Path[])new Path[]{this.location.resolve(Translog.getFilename(minGenerationToRecoverFrom - 1L)), this.location.resolve(Translog.getCommitCheckpointFileName(minGenerationToRecoverFrom - 1L))});
            Path commitCheckpoint = this.location.resolve(Translog.getCommitCheckpointFileName(checkpoint.generation));
            if (Files.exists(commitCheckpoint, new LinkOption[0])) {
                Checkpoint checkpointFromDisk = Checkpoint.read(commitCheckpoint);
                if (!checkpoint.equals(checkpointFromDisk)) {
                    throw new TranslogCorruptedException(commitCheckpoint.toString(), "checkpoint file " + String.valueOf(commitCheckpoint.getFileName()) + " already exists but has corrupted content: expected " + String.valueOf(checkpoint) + " but got " + String.valueOf(checkpointFromDisk));
                }
            } else {
                this.copyCheckpointTo(commitCheckpoint);
            }
            success = true;
        }
        finally {
            if (!success) {
                IOUtils.closeWhileHandlingException(foundTranslogs);
            }
        }
        return foundTranslogs;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void copyCheckpointTo(Path targetPath) throws IOException {
        Path tempFile = Files.createTempFile(this.location, TRANSLOG_FILE_PREFIX, CHECKPOINT_SUFFIX, new FileAttribute[0]);
        boolean tempFileRenamed = false;
        try {
            Files.copy(this.location.resolve(CHECKPOINT_FILE_NAME), tempFile, StandardCopyOption.REPLACE_EXISTING);
            IOUtils.fsync((Path)tempFile, (boolean)false);
            Files.move(tempFile, targetPath, StandardCopyOption.ATOMIC_MOVE);
            tempFileRenamed = true;
            IOUtils.fsync((Path)targetPath.getParent(), (boolean)true);
        }
        finally {
            if (!tempFileRenamed) {
                try {
                    Files.delete(tempFile);
                }
                catch (IOException ex) {
                    this.logger.warn(() -> new ParameterizedMessage("failed to delete temp file {}", (Object)tempFile), (Throwable)ex);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    TranslogReader openReader(Path path, Checkpoint checkpoint) throws IOException {
        FileChannel channel = FileChannel.open(path, StandardOpenOption.READ);
        try {
            assert (Translog.parseIdFromFileName(path) == checkpoint.generation) : "expected generation: " + Translog.parseIdFromFileName(path) + " but got: " + checkpoint.generation;
            TranslogReader reader = TranslogReader.open(channel, path, checkpoint, this.translogUUID);
            channel = null;
            TranslogReader translogReader = reader;
            return translogReader;
        }
        finally {
            IOUtils.close((Closeable)channel);
        }
    }

    public static long parseIdFromFileName(Path translogFile) {
        String fileName = translogFile.getFileName().toString();
        return Translog.parseIdFromFileName(fileName);
    }

    public static long parseIdFromFileName(String fileName) {
        Matcher matcher = PARSE_STRICT_ID_PATTERN.matcher(fileName);
        if (matcher.matches()) {
            try {
                return Long.parseLong(matcher.group(1));
            }
            catch (NumberFormatException e) {
                throw new IllegalStateException("number formatting issue in a file that passed PARSE_STRICT_ID_PATTERN: " + fileName + "]", e);
            }
        }
        throw new IllegalArgumentException("can't parse id from file: " + fileName);
    }

    public boolean isOpen() {
        return !this.closed.get();
    }

    protected static boolean calledFromOutsideOrViaTragedyClose() {
        List frames = Stream.of(Thread.currentThread().getStackTrace()).skip(3L).limit(10L).filter(f -> {
            try {
                return Translog.class.isAssignableFrom(Class.forName(f.getClassName()));
            }
            catch (Exception ignored) {
                return false;
            }
        }).collect(Collectors.toList());
        return frames.isEmpty() || frames.stream().anyMatch(f -> f.getMethodName().equals("closeOnTragicEvent"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        assert (Translog.calledFromOutsideOrViaTragedyClose()) : "Translog.close method is called from inside Translog, but not via closeOnTragicEvent method";
        if (this.closed.compareAndSet(false, true)) {
            try (ReleasableLock lock = this.writeLock.acquire();){
                try {
                    this.current.sync();
                }
                finally {
                    this.closeFilesIfNoPendingRetentionLocks();
                }
            }
            finally {
                this.logger.debug("translog closed");
            }
        }
    }

    public Path location() {
        return this.location;
    }

    public long currentFileGeneration() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            long l = this.current.getGeneration();
            return l;
        }
    }

    public long getMinFileGeneration() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            if (this.readers.isEmpty()) {
                long l = this.current.getGeneration();
                return l;
            }
            assert (this.readers.stream().map(BaseTranslogReader::getGeneration).min(Long::compareTo).get().equals(this.readers.get(0).getGeneration())) : "the first translog isn't the one with the minimum generation:" + String.valueOf(this.readers);
            long l = this.readers.get(0).getGeneration();
            return l;
        }
    }

    public int totalOperations() {
        return this.totalOperationsByMinGen(-1L);
    }

    public long sizeInBytes() {
        return this.sizeInBytesByMinGen(-1L);
    }

    long earliestLastModifiedAge() {
        ReleasableLock ignored = this.readLock.acquire();
        try {
            this.ensureOpen();
            long l = Translog.findEarliestLastModifiedAge(System.currentTimeMillis(), this.readers, this.current);
            if (ignored != null) {
                ignored.close();
            }
            return l;
        }
        catch (Throwable throwable) {
            try {
                if (ignored != null) {
                    try {
                        ignored.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new TranslogException(this.shardId, "Unable to get the earliest last modified time for the transaction log");
            }
        }
    }

    static long findEarliestLastModifiedAge(long currentTime, Iterable<TranslogReader> readers, TranslogWriter writer) throws IOException {
        long earliestTime = currentTime;
        for (BaseTranslogReader baseTranslogReader : readers) {
            earliestTime = Math.min(baseTranslogReader.getLastModifiedTime(), earliestTime);
        }
        return Math.max(0L, currentTime - Math.min(earliestTime, writer.getLastModifiedTime()));
    }

    public int totalOperationsByMinGen(long minGeneration) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            int n = Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(r -> r.getGeneration() >= minGeneration).mapToInt(rec$ -> rec$.totalOperations()).sum();
            return n;
        }
    }

    public int estimateTotalOperationsFromMinSeq(long minSeqNo) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            int n = this.readersAboveMinSeqNo(minSeqNo).mapToInt(BaseTranslogReader::totalOperations).sum();
            return n;
        }
    }

    public long sizeInBytesByMinGen(long minGeneration) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            long l = Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(r -> r.getGeneration() >= minGeneration).mapToLong(rec$ -> rec$.sizeInBytes()).sum();
            return l;
        }
    }

    TranslogWriter createWriter(long fileGeneration) throws IOException {
        TranslogWriter writer = this.createWriter(fileGeneration, this.getMinFileGeneration(), this.globalCheckpointSupplier.getAsLong(), this.persistedSequenceNumberConsumer);
        assert (writer.sizeInBytes() == (long)DEFAULT_HEADER_SIZE_IN_BYTES) : "Mismatch translog header size; empty translog size [" + writer.sizeInBytes() + ", header size [" + DEFAULT_HEADER_SIZE_IN_BYTES + "]";
        return writer;
    }

    TranslogWriter createWriter(long fileGeneration, long initialMinTranslogGen, long initialGlobalCheckpoint, LongConsumer persistedSequenceNumberConsumer) throws IOException {
        TranslogWriter newWriter;
        try {
            newWriter = TranslogWriter.create(this.shardId, this.translogUUID, fileGeneration, this.location.resolve(Translog.getFilename(fileGeneration)), this.getChannelFactory(), this.config.getBufferSize(), initialMinTranslogGen, initialGlobalCheckpoint, this.globalCheckpointSupplier, this::getMinFileGeneration, this.primaryTermSupplier.getAsLong(), this.tragedy, persistedSequenceNumberConsumer, this.bigArrays, this.indexSettings.isAssignedOnRemoteNode(), this.translogOperationHelper);
        }
        catch (IOException e) {
            throw new TranslogException(this.shardId, "failed to create new translog file", e);
        }
        return newWriter;
    }

    public Location add(Operation operation) throws IOException {
        ReleasableBytesStreamOutput out = new ReleasableBytesStreamOutput(this.bigArrays);
        try {
            long start = out.position();
            out.skip(4);
            Translog.writeOperationNoSize(new BufferedChecksumStreamOutput((StreamOutput)out), operation);
            long end = out.position();
            int operationSize = (int)(end - 4L - start);
            out.seek(start);
            out.writeInt(operationSize);
            out.seek(end);
            BytesReference bytes = out.bytes();
            ReleasableLock ignored = this.readLock.acquire();
            try {
                this.ensureOpen();
                if (operation.primaryTerm() > this.current.getPrimaryTerm()) {
                    assert (false) : "Operation term is newer than the current term; current term[" + this.current.getPrimaryTerm() + "], operation term[" + String.valueOf(operation) + "]";
                    throw new IllegalArgumentException("Operation term is newer than the current term; current term[" + this.current.getPrimaryTerm() + "], operation term[" + String.valueOf(operation) + "]");
                }
                Location location = this.current.add(bytes, operation.seqNo());
                if (ignored != null) {
                    ignored.close();
                }
                return location;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException | AlreadyClosedException ex) {
                    this.closeOnTragicEvent((Exception)ex);
                    throw ex;
                }
                catch (Exception ex) {
                    this.closeOnTragicEvent(ex);
                    throw new TranslogException(this.shardId, "Failed to write operation [" + String.valueOf(operation) + "]", ex);
                }
            }
        }
        finally {
            Releasables.close((Releasable)out);
        }
    }

    public boolean shouldRollGeneration() {
        long threshold = this.indexSettings.getGenerationThresholdSize().getBytes();
        try (ReleasableLock ignored = this.readLock.acquire();){
            boolean bl = this.current.sizeInBytes() > threshold;
            return bl;
        }
    }

    public Location getLastWriteLocation() {
        try (ReleasableLock lock = this.readLock.acquire();){
            Location location = new Location(this.current.generation, this.current.sizeInBytes() - 1L, Integer.MAX_VALUE);
            return location;
        }
    }

    public long getLastSyncedGlobalCheckpoint() {
        return this.getLastSyncedCheckpoint().globalCheckpoint;
    }

    final Checkpoint getLastSyncedCheckpoint() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            Checkpoint checkpoint = this.current.getLastSyncedCheckpoint();
            return checkpoint;
        }
    }

    public Snapshot newSnapshot() throws IOException {
        return this.newSnapshot(0L, Long.MAX_VALUE);
    }

    public Snapshot newSnapshot(long fromSeqNo, long toSeqNo) throws IOException {
        return this.newSnapshot(fromSeqNo, toSeqNo, false);
    }

    public Snapshot newSnapshot(long fromSeqNo, long toSeqNo, boolean requiredFullRange) throws IOException {
        assert (fromSeqNo <= toSeqNo) : fromSeqNo + " > " + toSeqNo;
        assert (fromSeqNo >= 0L) : "from_seq_no must be non-negative " + fromSeqNo;
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            TranslogSnapshot[] snapshots = (TranslogSnapshot[])Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(reader -> reader.getCheckpoint().minSeqNo <= toSeqNo && fromSeqNo <= reader.getCheckpoint().maxEffectiveSeqNo()).map(rec$ -> rec$.newSnapshot()).toArray(TranslogSnapshot[]::new);
            Snapshot snapshot = this.newMultiSnapshot(snapshots);
            SeqNoFilterSnapshot seqNoFilterSnapshot = new SeqNoFilterSnapshot(snapshot, fromSeqNo, toSeqNo, requiredFullRange);
            return seqNoFilterSnapshot;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Operation readOperation(Location location) throws IOException {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            if (location.generation < this.getMinFileGeneration()) {
                Operation operation = null;
                return operation;
            }
            if (this.current.generation == location.generation) {
                Operation operation = this.current.read(location);
                return operation;
            }
            int i = this.readers.size() - 1;
            while (i >= 0) {
                TranslogReader translogReader = this.readers.get(i);
                if (translogReader.generation == location.generation) {
                    Operation operation = translogReader.read(location);
                    return operation;
                }
                --i;
            }
            return null;
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Snapshot newMultiSnapshot(TranslogSnapshot[] snapshots) throws IOException {
        Closeable onClose;
        if (snapshots.length == 0) {
            onClose = () -> {};
        } else {
            assert (Arrays.stream(snapshots).map(BaseTranslogReader::getGeneration).min(Long::compareTo).get() == snapshots[0].generation) : "first reader generation of " + String.valueOf(snapshots) + " is not the smallest";
            onClose = this.acquireTranslogGenFromDeletionPolicy(snapshots[0].generation);
        }
        boolean success = false;
        try {
            MultiSnapshot result = new MultiSnapshot(snapshots, onClose);
            success = true;
            MultiSnapshot multiSnapshot = result;
            return multiSnapshot;
        }
        finally {
            if (!success) {
                onClose.close();
            }
        }
    }

    private Stream<? extends BaseTranslogReader> readersAboveMinSeqNo(long minSeqNo) {
        assert (this.readLock.isHeldByCurrentThread() || this.writeLock.isHeldByCurrentThread()) : "callers of readersAboveMinSeqNo must hold a lock: readLock [" + this.readLock.isHeldByCurrentThread() + "], writeLock [" + this.readLock.isHeldByCurrentThread() + "]";
        return Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(reader -> minSeqNo <= reader.getCheckpoint().maxEffectiveSeqNo());
    }

    public Closeable acquireRetentionLock() {
        try (ReleasableLock lock = this.readLock.acquire();){
            this.ensureOpen();
            long viewGen = this.getMinFileGeneration();
            Closeable closeable = this.acquireTranslogGenFromDeletionPolicy(viewGen);
            return closeable;
        }
    }

    private Closeable acquireTranslogGenFromDeletionPolicy(long viewGen) {
        Releasable toClose = this.deletionPolicy.acquireTranslogGen(viewGen);
        return () -> {
            try {
                toClose.close();
            }
            finally {
                this.trimUnreferencedReaders();
                this.closeFilesIfNoPendingRetentionLocks();
            }
        };
    }

    public void sync() throws IOException {
        try (ReleasableLock lock = this.readLock.acquire();){
            if (!this.closed.get()) {
                this.current.sync();
            }
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    public boolean syncNeeded() {
        try (ReleasableLock lock = this.readLock.acquire();){
            boolean bl = this.current.syncNeeded();
            return bl;
        }
    }

    public static String getFilename(long generation) {
        return TRANSLOG_FILE_PREFIX + generation + TRANSLOG_FILE_SUFFIX;
    }

    public static String getCommitCheckpointFileName(long generation) {
        return TRANSLOG_FILE_PREFIX + generation + CHECKPOINT_SUFFIX;
    }

    public void trimOperations(long belowTerm, long aboveSeqNo) throws IOException {
        assert (aboveSeqNo >= -1L) : "aboveSeqNo has to a valid sequence number";
        try (ReleasableLock lock = this.writeLock.acquire();){
            this.ensureOpen();
            if (this.current.getPrimaryTerm() < belowTerm) {
                throw new IllegalArgumentException("Trimming the translog can only be done for terms lower than the current one. Trim requested for term [ " + belowTerm + " ] , current is [ " + this.current.getPrimaryTerm() + " ]");
            }
            assert (this.current.assertNoSeqAbove(belowTerm, aboveSeqNo));
            ArrayList<TranslogReader> newReaders = new ArrayList<TranslogReader>(this.readers.size());
            try {
                for (TranslogReader reader : this.readers) {
                    TranslogReader newReader = reader.getPrimaryTerm() < belowTerm ? reader.closeIntoTrimmedReader(aboveSeqNo, this.getChannelFactory()) : reader;
                    newReaders.add(newReader);
                }
            }
            catch (IOException e) {
                IOUtils.closeWhileHandlingException(newReaders);
                this.tragedy.setTragicException(e);
                this.closeOnTragicEvent(e);
                throw e;
            }
            this.readers.clear();
            this.readers.addAll(newReaders);
        }
    }

    public abstract boolean ensureSynced(Location var1) throws IOException;

    public boolean ensureSynced(Stream<Location> locations) throws IOException {
        Optional<Location> max = locations.max(Location::compareTo);
        if (max.isPresent()) {
            return this.ensureSynced(max.get());
        }
        return false;
    }

    protected void closeOnTragicEvent(Exception ex) {
        assert (!this.readLock.isHeldByCurrentThread()) : Thread.currentThread().getName();
        if (this.tragedy.get() != null) {
            try {
                this.close();
            }
            catch (AlreadyClosedException alreadyClosedException) {
            }
            catch (Exception inner) {
                assert (ex != inner.getCause());
                ex.addSuppressed(inner);
            }
        }
    }

    public TranslogStats stats() {
        try (ReleasableLock lock = this.readLock.acquire();){
            long uncommittedGen = this.getMinGenerationForSeqNo((long)(this.deletionPolicy.getLocalCheckpointOfSafeCommit() + 1L)).translogFileGeneration;
            TranslogStats translogStats = new TranslogStats(this.totalOperations(), this.sizeInBytes(), this.totalOperationsByMinGen(uncommittedGen), this.sizeInBytesByMinGen(uncommittedGen), this.earliestLastModifiedAge());
            return translogStats;
        }
    }

    public TranslogConfig getConfig() {
        return this.config;
    }

    public TranslogDeletionPolicy getDeletionPolicy() {
        return this.deletionPolicy;
    }

    static void verifyChecksum(BufferedChecksumStreamInput in) throws IOException {
        long expectedChecksum = in.getChecksum();
        long readChecksum = Integer.toUnsignedLong(in.readInt());
        if (readChecksum != expectedChecksum) {
            throw new TranslogCorruptedException(in.getSource(), "checksum verification failed - expected: 0x" + Long.toHexString(expectedChecksum) + ", got: 0x" + Long.toHexString(readChecksum));
        }
    }

    public static List<Operation> readOperations(StreamInput input, String source) throws IOException {
        ArrayList<Operation> operations = new ArrayList<Operation>();
        int numOps = input.readInt();
        BufferedChecksumStreamInput checksumStreamInput = new BufferedChecksumStreamInput(input, source);
        for (int i = 0; i < numOps; ++i) {
            operations.add(Translog.readOperation(checksumStreamInput));
        }
        return operations;
    }

    static Operation readOperation(BufferedChecksumStreamInput in) throws IOException {
        Operation operation;
        try {
            int opSize = in.readInt();
            if (opSize < 4) {
                throw new TranslogCorruptedException(in.getSource(), "operation size must be at least 4 but was: " + opSize);
            }
            in.resetDigest();
            if (in.markSupported()) {
                in.mark(opSize);
                in.skip((long)(opSize - 4));
                Translog.verifyChecksum(in);
                in.reset();
            }
            operation = Operation.readOperation((StreamInput)in);
            Translog.verifyChecksum(in);
        }
        catch (EOFException e) {
            throw new TruncatedTranslogException(in.getSource(), "reached premature end of file, translog is truncated", e);
        }
        return operation;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void writeOperations(StreamOutput outStream, List<Operation> toWrite) throws IOException {
        ReleasableBytesStreamOutput out = new ReleasableBytesStreamOutput(BigArrays.NON_RECYCLING_INSTANCE);
        try {
            outStream.writeInt(toWrite.size());
            BufferedChecksumStreamOutput checksumStreamOutput = new BufferedChecksumStreamOutput((StreamOutput)out);
            for (Operation op : toWrite) {
                out.reset();
                long start = out.position();
                out.skip(4);
                Translog.writeOperationNoSize(checksumStreamOutput, op);
                long end = out.position();
                int operationSize = (int)(out.position() - 4L - start);
                out.seek(start);
                out.writeInt(operationSize);
                out.seek(end);
                out.bytes().writeTo((OutputStream)outStream);
            }
        }
        finally {
            Releasables.close((Releasable)out);
        }
    }

    public static void writeOperationNoSize(BufferedChecksumStreamOutput out, Operation op) throws IOException {
        out.resetDigest();
        Operation.writeOperation((StreamOutput)out, op);
        long checksum = out.getChecksum();
        out.writeInt((int)checksum);
    }

    public TranslogGeneration getMinGenerationForSeqNo(long seqNo) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            TranslogGeneration translogGeneration = new TranslogGeneration(this.translogUUID, Translog.minGenerationForSeqNo(seqNo, this.current, this.readers));
            return translogGeneration;
        }
    }

    static long minGenerationForSeqNo(long seqNo, TranslogWriter writer, List<TranslogReader> readers) {
        long minGen = writer.generation;
        for (TranslogReader reader : readers) {
            if (seqNo > reader.getCheckpoint().maxEffectiveSeqNo()) continue;
            minGen = Math.min(minGen, reader.getGeneration());
        }
        return minGen;
    }

    public void rollGeneration() throws IOException {
        this.syncBeforeRollGeneration();
        if (this.current.totalOperations() == 0 && this.primaryTermSupplier.getAsLong() == this.current.getPrimaryTerm()) {
            return;
        }
        try (ReleasableLock ignored = this.writeLock.acquire();){
            this.ensureOpen();
            try {
                TranslogReader reader = this.current.closeIntoReader();
                this.readers.add(reader);
                assert (Checkpoint.read((Path)this.location.resolve((String)CHECKPOINT_FILE_NAME)).generation == this.current.getGeneration());
                this.copyCheckpointTo(this.location.resolve(Translog.getCommitCheckpointFileName(this.current.getGeneration())));
                this.current = this.createWriter(this.current.getGeneration() + 1L);
                this.logger.trace("current translog set to [{}]", (Object)this.current.getGeneration());
            }
            catch (Exception e) {
                this.tragedy.setTragicException(e);
                this.closeOnTragicEvent(e);
                throw e;
            }
        }
    }

    void syncBeforeRollGeneration() throws IOException {
        this.sync();
    }

    public void trimUnreferencedReaders() throws IOException {
        try (ReleasableLock ignored = this.readLock.acquire();){
            if (this.closed.get()) {
                return;
            }
            if (this.getMinReferencedGen() == this.getMinFileGeneration()) {
                return;
            }
        }
        this.sync();
        try {
            ignored = this.writeLock.acquire();
            try {
                TranslogReader reader;
                if (this.closed.get()) {
                    return;
                }
                long minReferencedGen = this.getMinReferencedGen();
                Iterator<TranslogReader> iterator = this.readers.iterator();
                while (iterator.hasNext() && (reader = iterator.next()).getGeneration() < minReferencedGen) {
                    iterator.remove();
                    IOUtils.closeWhileHandlingException((Closeable)reader);
                    Path translogPath = reader.path();
                    this.logger.trace("delete translog file [{}], not referenced and not current anymore", (Object)translogPath);
                    this.current.sync();
                    this.deleteReaderFiles(reader);
                }
                assert (!this.readers.isEmpty() || this.current.generation == minReferencedGen) : "all readers were cleaned but the minReferenceGen [" + minReferencedGen + "] is not the current writer's gen [" + this.current.generation + "]";
            }
            finally {
                if (ignored != null) {
                    ignored.close();
                }
            }
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    protected long getMinReferencedGen() throws IOException {
        assert (this.readLock.isHeldByCurrentThread() || this.writeLock.isHeldByCurrentThread());
        long minReferencedGen = Math.min(this.deletionPolicy.minTranslogGenRequired(this.readers, this.current), Translog.minGenerationForSeqNo(this.deletionPolicy.getLocalCheckpointOfSafeCommit() + 1L, this.current, this.readers));
        assert (minReferencedGen >= this.getMinFileGeneration()) : "deletion policy requires a minReferenceGen of [" + minReferencedGen + "] but the lowest gen available is [" + this.getMinFileGeneration() + "]";
        assert (minReferencedGen <= this.currentFileGeneration()) : "deletion policy requires a minReferenceGen of [" + minReferencedGen + "] which is higher than the current generation [" + this.currentFileGeneration() + "]";
        return minReferencedGen;
    }

    protected void setMinSeqNoToKeep(long seqNo) {
    }

    protected void onDelete() {
    }

    abstract Releasable drainSync();

    void deleteReaderFiles(TranslogReader reader) {
        IOUtils.deleteFilesIgnoringExceptions((Path[])new Path[]{reader.path(), reader.path().resolveSibling(Translog.getCommitCheckpointFileName(reader.getGeneration()))});
    }

    void closeFilesIfNoPendingRetentionLocks() throws IOException {
        try (ReleasableLock ignored = this.writeLock.acquire();){
            if (this.closed.get() && this.deletionPolicy.pendingTranslogRefCount() == 0) {
                this.logger.trace("closing files. translog is closed and there are no pending retention locks");
                ArrayList<TranslogReader> toClose = new ArrayList<TranslogReader>(this.readers);
                toClose.add((TranslogReader)((Object)this.current));
                IOUtils.close(toClose);
            }
        }
    }

    public TranslogGeneration getGeneration() {
        return new TranslogGeneration(this.translogUUID, this.currentFileGeneration());
    }

    long getFirstOperationPosition() {
        return this.current.getFirstOperationOffset();
    }

    protected void ensureOpen() {
        if (this.closed.get()) {
            throw new AlreadyClosedException("translog is already closed", (Throwable)this.tragedy.get());
        }
    }

    ChannelFactory getChannelFactory() {
        return FileChannel::open;
    }

    public Exception getTragicException() {
        return this.tragedy.get();
    }

    static Checkpoint readCheckpoint(Path location) throws IOException {
        return Checkpoint.read(location.resolve(CHECKPOINT_FILE_NAME));
    }

    public static long readGlobalCheckpoint(Path location, String expectedTranslogUUID) throws IOException {
        Checkpoint checkpoint = Translog.readCheckpoint(location, expectedTranslogUUID);
        return checkpoint.globalCheckpoint;
    }

    private static Checkpoint readCheckpoint(Path location, String expectedTranslogUUID) throws IOException {
        Checkpoint checkpoint = Translog.readCheckpoint(location);
        Path translogFile = location.resolve(Translog.getFilename(checkpoint.generation));
        try (FileChannel channel = FileChannel.open(translogFile, StandardOpenOption.READ);){
            TranslogHeader.read(expectedTranslogUUID, translogFile, channel);
        }
        catch (TranslogCorruptedException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new TranslogCorruptedException(location.toString(), ex);
        }
        return checkpoint;
    }

    public static long readMinTranslogGeneration(Path location, String expectedTranslogUUID) throws IOException {
        Checkpoint checkpoint = Translog.readCheckpoint(location, expectedTranslogUUID);
        return checkpoint.minTranslogGeneration;
    }

    public String getTranslogUUID() {
        return this.translogUUID;
    }

    public long getMaxSeqNo() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            OptionalLong maxSeqNo = Stream.concat(this.readers.stream(), Stream.of(this.current)).mapToLong(reader -> reader.getCheckpoint().maxSeqNo).max();
            assert (maxSeqNo.isPresent()) : "must have at least one translog generation";
            long l = maxSeqNo.getAsLong();
            return l;
        }
    }

    TranslogWriter getCurrent() {
        return this.current;
    }

    List<TranslogReader> getReaders() {
        return this.readers;
    }

    public static String createEmptyTranslog(Path location, long initialGlobalCheckpoint, ShardId shardId, long primaryTerm) throws IOException {
        ChannelFactory channelFactory = FileChannel::open;
        return Translog.createEmptyTranslog(location, initialGlobalCheckpoint, shardId, channelFactory, primaryTerm);
    }

    static String createEmptyTranslog(Path location, long initialGlobalCheckpoint, ShardId shardId, ChannelFactory channelFactory, long primaryTerm) throws IOException {
        return Translog.createEmptyTranslog(location, shardId, initialGlobalCheckpoint, primaryTerm, null, channelFactory);
    }

    public static String createEmptyTranslog(Path location, ShardId shardId, long initialGlobalCheckpoint, long primaryTerm, @Nullable String translogUUID, @Nullable ChannelFactory factory) throws IOException {
        return Translog.createEmptyTranslog(location, shardId, initialGlobalCheckpoint, primaryTerm, translogUUID, factory, 1L);
    }

    public static String createEmptyTranslog(Path location, ShardId shardId, Checkpoint checkpoint) throws IOException {
        TranslogHeader translogHeader;
        Path highestGenTranslogFile = location.resolve(Translog.getFilename(checkpoint.generation));
        try (FileChannel channel = FileChannel.open(highestGenTranslogFile, StandardOpenOption.READ);){
            translogHeader = TranslogHeader.read(highestGenTranslogFile, channel);
        }
        String translogUUID = translogHeader.getTranslogUUID();
        long primaryTerm = translogHeader.getPrimaryTerm();
        ChannelFactory channelFactory = FileChannel::open;
        return Translog.createEmptyTranslog(location, shardId, -1L, primaryTerm, translogUUID, channelFactory, checkpoint.generation + 1L);
    }

    public static String createEmptyTranslog(Path location, ShardId shardId, long initialGlobalCheckpoint, long primaryTerm, @Nullable String translogUUID, @Nullable ChannelFactory factory, long generation) throws IOException {
        IOUtils.rm((Path[])new Path[]{location});
        Files.createDirectories(location, new FileAttribute[0]);
        ChannelFactory channelFactory = factory != null ? factory : FileChannel::open;
        String uuid = Strings.hasLength((String)translogUUID) ? translogUUID : UUIDs.randomBase64UUID();
        Path checkpointFile = location.resolve(CHECKPOINT_FILE_NAME);
        Path translogFile = location.resolve(Translog.getFilename(generation));
        Checkpoint checkpoint = Checkpoint.emptyTranslogCheckpoint(0L, generation, initialGlobalCheckpoint, generation);
        Checkpoint.write(channelFactory, checkpointFile, checkpoint, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
        TranslogWriter writer = TranslogWriter.create(shardId, uuid, generation, translogFile, channelFactory, TranslogConfig.EMPTY_TRANSLOG_BUFFER_SIZE, generation, initialGlobalCheckpoint, () -> {
            throw new UnsupportedOperationException();
        }, () -> {
            throw new UnsupportedOperationException();
        }, primaryTerm, new TragicExceptionHolder(), seqNo -> {
            throw new UnsupportedOperationException();
        }, BigArrays.NON_RECYCLING_INSTANCE, null, TranslogOperationHelper.DEFAULT);
        writer.close();
        return uuid;
    }

    public long getMinUnreferencedSeqNoInSegments(long minUnrefCheckpointInLastCommit) {
        return minUnrefCheckpointInLastCommit;
    }

    protected boolean shouldFlush() {
        return false;
    }

    @PublicApi(since="1.0.0")
    public static interface Operation {
        public Type opType();

        public long estimateSize();

        public Source getSource();

        public long seqNo();

        public long primaryTerm();

        public static Operation readOperation(StreamInput input) throws IOException {
            Type type = Type.fromId(input.readByte());
            switch (type.ordinal()) {
                case 0: 
                case 1: {
                    return new Index(input);
                }
                case 2: {
                    return new Delete(input);
                }
                case 3: {
                    return new NoOp(input);
                }
            }
            throw new AssertionError((Object)("no case for [" + String.valueOf((Object)type) + "]"));
        }

        public static void writeOperation(StreamOutput output, Operation operation) throws IOException {
            output.writeByte(operation.opType().id());
            switch (operation.opType().ordinal()) {
                case 0: 
                case 1: {
                    ((Index)operation).write(output);
                    break;
                }
                case 2: {
                    ((Delete)operation).write(output);
                    break;
                }
                case 3: {
                    ((NoOp)operation).write(output);
                    break;
                }
                default: {
                    throw new AssertionError((Object)("no case for [" + String.valueOf((Object)operation.opType()) + "]"));
                }
            }
        }

        @PublicApi(since="1.0.0")
        public static enum Type {
            CREATE(1),
            INDEX(2),
            DELETE(3),
            NO_OP(4);

            private final byte id;

            private Type(byte id) {
                this.id = id;
            }

            public byte id() {
                return this.id;
            }

            public static Type fromId(byte id) {
                switch (id) {
                    case 1: {
                        return CREATE;
                    }
                    case 2: {
                        return INDEX;
                    }
                    case 3: {
                        return DELETE;
                    }
                    case 4: {
                        return NO_OP;
                    }
                }
                throw new IllegalArgumentException("no type mapped for [" + id + "]");
            }
        }
    }

    @PublicApi(since="1.0.0")
    public static class Location
    implements Comparable<Location> {
        public final long generation;
        public final long translogLocation;
        public final int size;

        public Location(long generation, long translogLocation, int size) {
            this.generation = generation;
            this.translogLocation = translogLocation;
            this.size = size;
        }

        public String toString() {
            return "[generation: " + this.generation + ", location: " + this.translogLocation + ", size: " + this.size + "]";
        }

        @Override
        public int compareTo(Location o) {
            if (this.generation == o.generation) {
                return Long.compare(this.translogLocation, o.translogLocation);
            }
            return Long.compare(this.generation, o.generation);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Location location = (Location)o;
            if (this.generation != location.generation) {
                return false;
            }
            if (this.translogLocation != location.translogLocation) {
                return false;
            }
            return this.size == location.size;
        }

        public int hashCode() {
            int result = Long.hashCode(this.generation);
            result = 31 * result + Long.hashCode(this.translogLocation);
            result = 31 * result + this.size;
            return result;
        }
    }

    @PublicApi(since="1.0.0")
    public static interface Snapshot
    extends Closeable {
        public int totalOperations();

        default public int skippedOperations() {
            return 0;
        }

        public Operation next() throws IOException;
    }

    private static final class SeqNoFilterSnapshot
    implements Snapshot {
        private final Snapshot delegate;
        private int filteredOpsCount;
        private int opsCount;
        private boolean requiredFullRange;
        private final long fromSeqNo;
        private final long toSeqNo;

        SeqNoFilterSnapshot(Snapshot delegate, long fromSeqNo, long toSeqNo, boolean requiredFullRange) {
            assert (fromSeqNo <= toSeqNo) : "from_seq_no[" + fromSeqNo + "] > to_seq_no[" + toSeqNo + "]";
            this.delegate = delegate;
            this.fromSeqNo = fromSeqNo;
            this.toSeqNo = toSeqNo;
            this.requiredFullRange = requiredFullRange;
        }

        @Override
        public int totalOperations() {
            return this.delegate.totalOperations();
        }

        @Override
        public int skippedOperations() {
            return this.filteredOpsCount + this.delegate.skippedOperations();
        }

        @Override
        public Operation next() throws IOException, MissingHistoryOperationsException {
            Operation op;
            while ((op = this.delegate.next()) != null) {
                if (this.fromSeqNo <= op.seqNo() && op.seqNo() <= this.toSeqNo) {
                    ++this.opsCount;
                    return op;
                }
                ++this.filteredOpsCount;
            }
            if (this.requiredFullRange && this.toSeqNo - this.fromSeqNo + 1L != (long)this.opsCount) {
                throw new MissingHistoryOperationsException("Not all operations between from_seqno [" + this.fromSeqNo + "] and to_seqno [" + this.toSeqNo + "] found");
            }
            return null;
        }

        @Override
        public void close() throws IOException {
            this.delegate.close();
        }
    }

    @PublicApi(since="1.0.0")
    public static final class TranslogGeneration {
        public final String translogUUID;
        public final long translogFileGeneration;

        public TranslogGeneration(String translogUUID, long translogFileGeneration) {
            this.translogUUID = translogUUID;
            this.translogFileGeneration = translogFileGeneration;
        }
    }

    @PublicApi(since="1.0.0")
    public static enum Durability {
        ASYNC,
        REQUEST;

    }

    public static class NoOp
    implements Operation {
        private final long seqNo;
        private final long primaryTerm;
        private final String reason;

        @Override
        public long seqNo() {
            return this.seqNo;
        }

        @Override
        public long primaryTerm() {
            return this.primaryTerm;
        }

        public String reason() {
            return this.reason;
        }

        private NoOp(StreamInput in) throws IOException {
            this.seqNo = in.readLong();
            this.primaryTerm = in.readLong();
            this.reason = in.readString();
        }

        public NoOp(long seqNo, long primaryTerm, String reason) {
            assert (seqNo > -1L);
            assert (primaryTerm >= 0L);
            assert (reason != null);
            this.seqNo = seqNo;
            this.primaryTerm = primaryTerm;
            this.reason = reason;
        }

        private void write(StreamOutput out) throws IOException {
            out.writeLong(this.seqNo);
            out.writeLong(this.primaryTerm);
            out.writeString(this.reason);
        }

        @Override
        public Operation.Type opType() {
            return Operation.Type.NO_OP;
        }

        @Override
        public long estimateSize() {
            return 2 * this.reason.length() + 16;
        }

        @Override
        public Source getSource() {
            throw new UnsupportedOperationException("source does not exist for a no-op");
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            NoOp that = (NoOp)obj;
            return this.seqNo == that.seqNo && this.primaryTerm == that.primaryTerm && this.reason.equals(that.reason);
        }

        public int hashCode() {
            return 961 * Long.hashCode(this.seqNo) + 31 * Long.hashCode(this.primaryTerm) + this.reason().hashCode();
        }

        public String toString() {
            return "NoOp{seqNo=" + this.seqNo + ", primaryTerm=" + this.primaryTerm + ", reason='" + this.reason + "'}";
        }
    }

    public static class Delete
    implements Operation {
        private static final int FORMAT_6_0 = 4;
        public static final int FORMAT_NO_PARENT = 5;
        public static final int FORMAT_NO_VERSION_TYPE = 6;
        public static final int FORMAT_NO_DOC_TYPE = 7;
        public static final int SERIALIZATION_FORMAT = 7;
        private final String id;
        private final long seqNo;
        private final long primaryTerm;
        private final long version;

        private Delete(StreamInput in) throws IOException {
            int format = in.readVInt();
            assert (format >= 4) : "format was: " + format;
            if (format < 7) {
                in.readString();
            }
            this.id = in.readString();
            if (format < 7) {
                String docType = in.readString();
                assert (docType.equals("_id")) : docType + " != _id";
                in.readBytesRef();
            }
            this.version = in.readLong();
            if (format < 6) {
                in.readByte();
            }
            this.seqNo = in.readLong();
            this.primaryTerm = in.readLong();
        }

        public Delete(Engine.Delete delete, Engine.DeleteResult deleteResult) {
            this(delete.id(), deleteResult.getSeqNo(), delete.primaryTerm(), deleteResult.getVersion());
        }

        public Delete(String id, long seqNo, long primaryTerm) {
            this(id, seqNo, primaryTerm, -3L);
        }

        public Delete(String id, long seqNo, long primaryTerm, long version) {
            this.id = Objects.requireNonNull(id);
            this.seqNo = seqNo;
            this.primaryTerm = primaryTerm;
            this.version = version;
        }

        @Override
        public Operation.Type opType() {
            return Operation.Type.DELETE;
        }

        @Override
        public long estimateSize() {
            return this.id.length() * 2 + 24;
        }

        public String id() {
            return this.id;
        }

        @Override
        public long seqNo() {
            return this.seqNo;
        }

        @Override
        public long primaryTerm() {
            return this.primaryTerm;
        }

        public long version() {
            return this.version;
        }

        @Override
        public Source getSource() {
            throw new IllegalStateException("trying to read doc source from delete operation");
        }

        private void write(StreamOutput out) throws IOException {
            int format = out.getVersion().onOrAfter(Version.V_2_0_0) ? 7 : 6;
            out.writeVInt(format);
            if (format < 7) {
                out.writeString("_doc");
            }
            out.writeString(this.id);
            if (format < 7) {
                out.writeString("_id");
                out.writeBytesRef(Uid.encodeId(this.id));
            }
            out.writeLong(this.version);
            if (format < 6) {
                out.writeByte(VersionType.EXTERNAL.getValue());
            }
            out.writeLong(this.seqNo);
            out.writeLong(this.primaryTerm);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Delete delete = (Delete)o;
            return this.version == delete.version && this.seqNo == delete.seqNo && this.primaryTerm == delete.primaryTerm;
        }

        public int hashCode() {
            int result = Long.hashCode(this.seqNo);
            result = 31 * result + Long.hashCode(this.primaryTerm);
            result = 31 * result + Long.hashCode(this.version);
            return result;
        }

        public String toString() {
            return "Delete{seqNo=" + this.seqNo + ", primaryTerm=" + this.primaryTerm + ", version=" + this.version + "}";
        }
    }

    @PublicApi(since="1.0.0")
    public static class Index
    implements Operation {
        public static final int FORMAT_6_0 = 8;
        public static final int FORMAT_NO_PARENT = 9;
        public static final int FORMAT_NO_VERSION_TYPE = 10;
        public static final int FORMAT_NO_DOC_TYPE = 11;
        public static final int SERIALIZATION_FORMAT = 11;
        private final String id;
        private final long autoGeneratedIdTimestamp;
        private final long seqNo;
        private final long primaryTerm;
        private final long version;
        private final BytesReference source;
        private final String routing;

        private Index(StreamInput in) throws IOException {
            int format = in.readVInt();
            assert (format >= 8) : "format was: " + format;
            this.id = in.readString();
            if (format < 11) {
                in.readString();
            }
            this.source = in.readBytesReference();
            this.routing = in.readOptionalString();
            if (format < 9) {
                in.readOptionalString();
            }
            this.version = in.readLong();
            if (format < 10) {
                in.readByte();
            }
            this.autoGeneratedIdTimestamp = in.readLong();
            this.seqNo = in.readLong();
            this.primaryTerm = in.readLong();
        }

        public Index(Engine.Index index, Engine.IndexResult indexResult) {
            this.id = index.id();
            this.source = index.source();
            this.routing = index.routing();
            this.seqNo = indexResult.getSeqNo();
            this.primaryTerm = index.primaryTerm();
            this.version = indexResult.getVersion();
            this.autoGeneratedIdTimestamp = index.getAutoGeneratedIdTimestamp();
        }

        public Index(String id, long seqNo, long primaryTerm, byte[] source) {
            this(id, seqNo, primaryTerm, -3L, source, null, -1L);
        }

        public Index(String id, long seqNo, long primaryTerm, long version, byte[] source, String routing, long autoGeneratedIdTimestamp) {
            this.id = id;
            this.source = new BytesArray(source);
            this.seqNo = seqNo;
            this.primaryTerm = primaryTerm;
            this.version = version;
            this.routing = routing;
            this.autoGeneratedIdTimestamp = autoGeneratedIdTimestamp;
        }

        @Override
        public Operation.Type opType() {
            return Operation.Type.INDEX;
        }

        @Override
        public long estimateSize() {
            return 2 * this.id.length() + this.source.length() + (this.routing != null ? 2 * this.routing.length() : 0) + 32;
        }

        public String id() {
            return this.id;
        }

        public String routing() {
            return this.routing;
        }

        public BytesReference source() {
            return this.source;
        }

        @Override
        public long seqNo() {
            return this.seqNo;
        }

        @Override
        public long primaryTerm() {
            return this.primaryTerm;
        }

        public long version() {
            return this.version;
        }

        @Override
        public Source getSource() {
            return new Source(this.source, this.routing);
        }

        private void write(StreamOutput out) throws IOException {
            int format = out.getVersion().onOrAfter(Version.V_2_0_0) ? 11 : 10;
            out.writeVInt(format);
            out.writeString(this.id);
            if (format < 11) {
                out.writeString("_doc");
            }
            out.writeBytesReference(this.source);
            out.writeOptionalString(this.routing);
            if (format < 9) {
                out.writeOptionalString(null);
            }
            out.writeLong(this.version);
            if (format < 10) {
                out.writeByte(VersionType.EXTERNAL.getValue());
            }
            out.writeLong(this.autoGeneratedIdTimestamp);
            out.writeLong(this.seqNo);
            out.writeLong(this.primaryTerm);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Index index = (Index)o;
            if (this.version != index.version || this.seqNo != index.seqNo || this.primaryTerm != index.primaryTerm || !this.id.equals(index.id) || this.autoGeneratedIdTimestamp != index.autoGeneratedIdTimestamp || !this.source.equals((Object)index.source)) {
                return false;
            }
            return !(this.routing != null ? !this.routing.equals(index.routing) : index.routing != null);
        }

        public int hashCode() {
            int result = this.id.hashCode();
            result = 31 * result + Long.hashCode(this.seqNo);
            result = 31 * result + Long.hashCode(this.primaryTerm);
            result = 31 * result + Long.hashCode(this.version);
            result = 31 * result + this.source.hashCode();
            result = 31 * result + (this.routing != null ? this.routing.hashCode() : 0);
            result = 31 * result + Long.hashCode(this.autoGeneratedIdTimestamp);
            return result;
        }

        public String toString() {
            return "Index{id='" + this.id + "', seqNo=" + this.seqNo + ", primaryTerm=" + this.primaryTerm + ", version=" + this.version + ", autoGeneratedIdTimestamp=" + this.autoGeneratedIdTimestamp + "}";
        }

        public long getAutoGeneratedIdTimestamp() {
            return this.autoGeneratedIdTimestamp;
        }
    }

    @PublicApi(since="1.0.0")
    public static class Source {
        public final BytesReference source;
        public final String routing;

        public Source(BytesReference source, String routing) {
            this.source = source;
            this.routing = routing;
        }
    }
}

