/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.deployment;

import de.virtimo.bpc.api.BpcServicesTracker;
import de.virtimo.bpc.api.Checker;
import de.virtimo.bpc.api.CoreBundleConfiguration;
import de.virtimo.bpc.api.ModuleManager;
import de.virtimo.bpc.api.SystemException;
import de.virtimo.bpc.api.deployment.DeploymentManager;
import de.virtimo.bpc.api.deployment.resource.DeploymentData;
import de.virtimo.bpc.api.exception.ModuleNotFoundException;
import de.virtimo.bpc.api.exception.ServiceNotFoundException;
import de.virtimo.bpc.core.CoreModule;
import de.virtimo.bpc.core.deployment.DeploymentImporterSession;
import de.virtimo.bpc.core.deployment.DeploymentInitiatorImpl;
import de.virtimo.bpc.core.deployment.resource.DeploymentDataImpl;
import de.virtimo.bpc.core.utils.FelixFileinstallUtil;
import de.virtimo.bpc.util.JsonUtil;
import de.virtimo.bpc.util.ThreadFactoryWithNamePrefix;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;

public class DeploymentImporter
implements Checker {
    private static final Logger LOGGER = LogManager.getLogger(DeploymentImporter.class);
    private static final int DEPLOYMENT_FOLDER_CHECK_RUNS_EVERY_SECONDS = 3;
    private static final String DEPLOYMENT_FILE_EXTENSION = ".bpc-deployment-export";
    private static final String DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_INVALID = ".invalid_json";
    private static final String DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_EMPTY = ".empty";
    private static final String DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_ERROR = ".error";
    private static final String DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_ERROR_ON_IMPORT = ".error_on_import";
    private static final String DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_PROCESSED = ".processed";
    private ScheduledExecutorService executorService;
    private ScheduledFuture<?> checkerHandle;
    private final BundleContext bundleContext;
    private BpcServicesTracker<ModuleManager> moduleManagerTracker;
    private BpcServicesTracker<DeploymentManager> deploymentManagerTracker;
    private BpcServicesTracker<CoreBundleConfiguration> coreBundleConfigurationTracker;
    private boolean importIsAlreadyRunning = false;

    public DeploymentImporter(BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    @Override
    public void startChecker() {
        LOGGER.info("startChecker");
        if (this.checkerHandle != null) {
            LOGGER.warn("Deployment importer is already running");
            return;
        }
        BpcServicesTracker.stopAll(this);
        this.moduleManagerTracker = new BpcServicesTracker<ModuleManager>(this.bundleContext, ModuleManager.class);
        this.deploymentManagerTracker = new BpcServicesTracker<DeploymentManager>(this.bundleContext, DeploymentManager.class);
        this.coreBundleConfigurationTracker = new BpcServicesTracker<CoreBundleConfiguration>(this.bundleContext, CoreBundleConfiguration.class);
        try {
            this.executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryWithNamePrefix("bpc-core-deployment-importer"));
            this.checkerHandle = this.executorService.scheduleWithFixedDelay(this.createCheckerRunnable(), 0L, 3L, TimeUnit.SECONDS);
        }
        catch (Exception ex) {
            LOGGER.error("Failed to start the deployment importer.", (Throwable)ex);
        }
    }

    @Override
    public void stopChecker() {
        LOGGER.info("stopChecker");
        if (this.checkerHandle != null) {
            if (this.checkerHandle.cancel(true)) {
                LOGGER.info("Running deployment importer cancelled");
            } else {
                LOGGER.warn("Could not cancel the running deployment importer");
            }
            this.checkerHandle = null;
        }
        if (this.executorService != null) {
            this.executorService.shutdownNow();
            this.executorService = null;
        }
        BpcServicesTracker.stopAll(this);
    }

    private Runnable createCheckerRunnable() {
        LOGGER.info("createCheckerRunnable");
        return new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                if (!DeploymentImporter.this.isThisServerAlreadyDefinedAsMasterOrSlave()) {
                    LOGGER.info("Skipping ... no master/slave server defined yet. A master server is necessary to make backups of the bpc-configuration index before importing the deployment export.");
                    return;
                }
                if (DeploymentImporter.this.importIsAlreadyRunning) {
                    LOGGER.info("Skipping ... a deployment importer is already running.");
                    return;
                }
                DeploymentImporter.this.importIsAlreadyRunning = true;
                try {
                    File[] deploymentExportFiles = DeploymentImporter.this.getDeploymentExportFilesFromKarafDeployFolder();
                    if (deploymentExportFiles != null) {
                        for (File deploymentExportFile : deploymentExportFiles) {
                            if (!DeploymentImporter.this.canImportFile(deploymentExportFile)) continue;
                            try {
                                DeploymentImporter.this.importFile(deploymentExportFile);
                                DeploymentImporter.this.renameFile(deploymentExportFile, DeploymentImporter.DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_PROCESSED);
                            }
                            catch (Exception ex) {
                                LOGGER.error("Deployment import of file '" + String.valueOf(deploymentExportFile) + "' failed.", (Throwable)ex);
                                DeploymentImporter.this.renameFile(deploymentExportFile, DeploymentImporter.DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_ERROR_ON_IMPORT);
                            }
                        }
                    }
                }
                catch (Throwable t) {
                    LOGGER.error("Failed to perform the deployment importer.", t);
                }
                finally {
                    DeploymentImporter.this.importIsAlreadyRunning = false;
                }
            }
        };
    }

    private boolean isThisServerAlreadyDefinedAsMasterOrSlave() {
        LOGGER.info("isThisServerAlreadyDefinedAsMasterOrSlave");
        try {
            return this.moduleManagerTracker.getService().getModuleByClass(CoreModule.class).isMasterAndSlaveServerAlreadyDefined();
        }
        catch (ServiceNotFoundException e) {
            return false;
        }
        catch (ModuleNotFoundException ex) {
            return false;
        }
    }

    private File[] getDeploymentExportFilesFromKarafDeployFolder() throws ServiceNotFoundException, InvalidSyntaxException, IOException {
        LOGGER.info("getDeploymentExportFilesFromKarafDeployFolder");
        File deployFolder = FelixFileinstallUtil.getDeployFolder(this.bundleContext);
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File f, String name) {
                return name.endsWith(DeploymentImporter.DEPLOYMENT_FILE_EXTENSION);
            }
        };
        return deployFolder.listFiles(filter);
    }

    private boolean renameFile(File deploymentExportFile, String fileExtensionPostfix) {
        LOGGER.info("renameFile deploymentExportFile={}, fileExtensionPostfix={}", (Object)deploymentExportFile, (Object)fileExtensionPostfix);
        File processedDeploymentExportFile = new File(deploymentExportFile.getAbsolutePath() + fileExtensionPostfix);
        return deploymentExportFile.renameTo(processedDeploymentExportFile);
    }

    private boolean canImportFile(File deploymentExportFile) {
        DeploymentData deploymentData;
        LOGGER.info("canImportFile deploymentExportFile={}", (Object)deploymentExportFile);
        try {
            deploymentData = JsonUtil.getInstance().convertJsonFileToPojo(deploymentExportFile, DeploymentDataImpl.class);
            if (!deploymentData.hasSettingsToDeploy()) {
                LOGGER.info("The given deployment file '{}' does not contain any settings related data. Renaming the file so that it does not get processed again.", (Object)deploymentExportFile);
                this.renameFile(deploymentExportFile, DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_EMPTY);
                return false;
            }
        }
        catch (Exception ex) {
            LOGGER.error("Failed to check if the file '" + String.valueOf(deploymentExportFile) + "' can be used for deployment importer. Renaming the file so that it does not get processed again.", (Throwable)ex);
            this.renameFile(deploymentExportFile, DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_INVALID);
            return false;
        }
        try {
            ModuleManager moduleManager = this.moduleManagerTracker.getService();
            if (deploymentData.hasSettingsToDeploy()) {
                Set<String> relatedModuleIDs = deploymentData.getSettingsToDeploy().getRelatedModuleIDs();
                for (String relatedModuleID : relatedModuleIDs) {
                    LOGGER.info("Checking if the module '{}' has been loaded by the module manager.", (Object)relatedModuleID);
                    try {
                        moduleManager.getModuleById(relatedModuleID);
                    }
                    catch (ModuleNotFoundException ex) {
                        LOGGER.warn("Deployment importer cannot run with missing (not yet loaded) module '{}'. Will try again later ...", (Object)relatedModuleID);
                        return false;
                    }
                }
            }
            return true;
        }
        catch (Exception ex) {
            LOGGER.error("Failed to check if the file '" + String.valueOf(deploymentExportFile) + "' can be used for deployment importer.", (Throwable)ex);
            this.renameFile(deploymentExportFile, DEPLOYMENT_FILE_EXTENSION_POSTFIX_WHEN_ERROR);
            return false;
        }
    }

    private void importFile(File deploymentExportFile) throws SystemException, IOException {
        LOGGER.info("importFile deploymentExportFile={}", (Object)deploymentExportFile);
        DeploymentManager deploymentManager = this.deploymentManagerTracker.getService();
        CoreBundleConfiguration coreBundleConfiguration = this.coreBundleConfigurationTracker.getService();
        DeploymentImporterSession userSession = new DeploymentImporterSession();
        DeploymentInitiatorImpl deploymentInitiator = new DeploymentInitiatorImpl("from-deploy-folder", coreBundleConfiguration.getBpcName(), coreBundleConfiguration.getServerUUID());
        DeploymentData deploymentData = JsonUtil.getInstance().convertJsonFileToPojo(deploymentExportFile, DeploymentDataImpl.class);
        deploymentManager.importDeployment(userSession, deploymentInitiator, deploymentData, false, true);
    }
}

