/*
 * Decompiled with CFR 0.152.
 */
package de.virtimo.bpc.core.notification;

import com.fasterxml.jackson.core.JsonProcessingException;
import de.virtimo.bpc.api.ErrorCode;
import de.virtimo.bpc.api.auth.Organisation;
import de.virtimo.bpc.api.auth.Role;
import de.virtimo.bpc.api.exception.OpenSearchIndexNotFoundException;
import de.virtimo.bpc.api.exception.OpenSearchRelatedException;
import de.virtimo.bpc.api.opensearch.BpcIndexCreateCallable;
import de.virtimo.bpc.api.opensearch.BpcIndexState;
import de.virtimo.bpc.api.service.OpenSearchService;
import de.virtimo.bpc.core.exception.CoreErrorCode;
import de.virtimo.bpc.core.notification.Notification;
import de.virtimo.bpc.core.notification.NotificationData;
import de.virtimo.bpc.core.notification.NotificationException;
import de.virtimo.bpc.core.notification.NotificationNotFoundException;
import de.virtimo.bpc.core.notification.Notifications;
import de.virtimo.bpc.util.JsonUtil;
import de.virtimo.bpc.util.MapUtil;
import de.virtimo.bpc.util.StringUtil;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.bulk.BulkRequest;
import org.opensearch.action.bulk.BulkResponse;
import org.opensearch.action.delete.DeleteRequest;
import org.opensearch.action.delete.DeleteResponse;
import org.opensearch.action.get.GetRequest;
import org.opensearch.action.get.GetResponse;
import org.opensearch.action.get.MultiGetItemResponse;
import org.opensearch.action.get.MultiGetRequest;
import org.opensearch.action.get.MultiGetResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.index.IndexResponse;
import org.opensearch.action.search.SearchRequest;
import org.opensearch.action.search.SearchResponse;
import org.opensearch.action.support.WriteRequest;
import org.opensearch.action.update.UpdateRequest;
import org.opensearch.client.RequestOptions;
import org.opensearch.common.xcontent.XContentType;
import org.opensearch.core.rest.RestStatus;
import org.opensearch.core.xcontent.MediaType;
import org.opensearch.index.query.BoolQueryBuilder;
import org.opensearch.index.query.ExistsQueryBuilder;
import org.opensearch.index.query.QueryBuilder;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.index.query.RangeQueryBuilder;
import org.opensearch.search.SearchHit;
import org.opensearch.search.builder.SearchSourceBuilder;
import org.opensearch.search.sort.SortOrder;

public class NotificationPersistenceHandler {
    private static final Logger LOGGER = LogManager.getLogger(NotificationPersistenceHandler.class);
    public static final String NOTIFICATION_INDEX_ALIAS_NAME = "bpc-notification";
    private final OpenSearchService oss;

    public NotificationPersistenceHandler(OpenSearchService oss) {
        this.oss = oss;
    }

    public void createBpcNotificationIndexIfMissing() throws NotificationException {
        LOGGER.debug("createBpcNotificationIndexIfMissing");
        try {
            BpcIndexState bpcNotificationIndexState = this.oss.getIndexState(NOTIFICATION_INDEX_ALIAS_NAME);
            bpcNotificationIndexState.prepareUsing(new BpcIndexCreateCallable(){

                @Override
                public String createIndex(OpenSearchService oss) throws OpenSearchRelatedException {
                    return oss.getManagedIndicesHandler().createManagedIndex(NotificationPersistenceHandler.NOTIFICATION_INDEX_ALIAS_NAME);
                }
            });
        }
        catch (Exception ex) {
            throw new NotificationException((ErrorCode)CoreErrorCode.NOTIFICATION_FAILED, "Failed to prepare the notification index '${index}'.", MapUtil.mapOf("index", NOTIFICATION_INDEX_ALIAS_NAME), (Throwable)ex);
        }
    }

    public void deleteNotification(String notificationId) throws NotificationNotFoundException, OpenSearchRelatedException, OpenSearchIndexNotFoundException {
        LOGGER.info("deleteNotification notificationId={}", (Object)notificationId);
        DeleteRequest deleteRequest = ((DeleteRequest)new DeleteRequest().index(NOTIFICATION_INDEX_ALIAS_NAME)).id(notificationId);
        DeleteResponse response = this.oss.delete(deleteRequest, RequestOptions.DEFAULT);
        if (response == null || response.status() == RestStatus.NOT_FOUND) {
            throw new NotificationNotFoundException(notificationId);
        }
    }

    public void deleteNotifications(List<String> notificationIds) throws NotificationNotFoundException, OpenSearchRelatedException, OpenSearchIndexNotFoundException {
        LOGGER.info("deleteNotifications notificationIds={}", notificationIds);
        BulkRequest bulkDeleteRequest = new BulkRequest();
        for (String notificationId : notificationIds) {
            bulkDeleteRequest.add(((DeleteRequest)new DeleteRequest().index(NOTIFICATION_INDEX_ALIAS_NAME)).id(notificationId));
        }
        BulkResponse br = this.oss.bulk(bulkDeleteRequest, RequestOptions.DEFAULT);
        if (br.hasFailures()) {
            LOGGER.warn("deleteNotifications - bulk response has failures: {}", (Object)br.buildFailureMessage());
        }
    }

    public Notification loadNotification(String notificationId) throws NotificationNotFoundException, OpenSearchRelatedException, OpenSearchIndexNotFoundException, NotificationException {
        LOGGER.info("loadNotification notificationId={}", (Object)notificationId);
        GetRequest getRequest = ((GetRequest)new GetRequest().index(NOTIFICATION_INDEX_ALIAS_NAME)).id(notificationId);
        GetResponse response = this.oss.get(getRequest, RequestOptions.DEFAULT);
        if (!response.isExists()) {
            throw new NotificationNotFoundException(notificationId);
        }
        return this.createNotificationFromOpenSearchSource(response.getVersion(), response.getSourceAsString());
    }

    public Notification saveNotification(Notification notification) throws NotificationException, OpenSearchRelatedException, OpenSearchIndexNotFoundException {
        LOGGER.info("saveNotification notification=...");
        try {
            IndexRequest indexRequest = (IndexRequest)((IndexRequest)new IndexRequest().index(NOTIFICATION_INDEX_ALIAS_NAME)).id(notification.getId()).source(JsonUtil.getInstance().convertPojoToJsonString(notification), (MediaType)XContentType.JSON).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
            IndexResponse response = this.oss.index(indexRequest, RequestOptions.DEFAULT);
            return new Notification.Builder(notification).version(response.getVersion()).build();
        }
        catch (JsonProcessingException ex) {
            throw new NotificationException((ErrorCode)CoreErrorCode.NOTIFICATION_FAILED, "Failed to convert the notification object with the ID '${id}' to a JSON string.", MapUtil.mapOf("id", notification.getId()), (Throwable)ex);
        }
    }

    public void bulkUpdateNotificationReadStatus(List<Notification> notifications) throws NotificationException, OpenSearchRelatedException, OpenSearchIndexNotFoundException {
        LOGGER.info("bulkUpdateNotificationReadStatus notifications=...");
        BulkRequest bulkRequest = new BulkRequest();
        for (Notification notification : notifications) {
            Map<String, Object> jsonUpdateMap = MapUtil.mapOf("readByUserIds", notification.getReadByUserIds());
            bulkRequest.add(new UpdateRequest(NOTIFICATION_INDEX_ALIAS_NAME, notification.getId()).doc(jsonUpdateMap, (MediaType)XContentType.JSON)).setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
        }
        BulkResponse br = this.oss.bulk(bulkRequest, RequestOptions.DEFAULT);
        if (br.hasFailures()) {
            LOGGER.warn("bulkUpdateNotificationReadStatus - bulk response has failures: {}", (Object)br.buildFailureMessage());
        }
    }

    public Notifications loadNotifications(Integer start, Integer limit, String userName, List<Role> roles, List<Organisation> organisations) throws OpenSearchRelatedException, OpenSearchIndexNotFoundException {
        LOGGER.info("loadNotifications start={}, limit={}, userName={}, roles={}, organisations={}", (Object)start, (Object)limit, (Object)userName, roles, organisations);
        SearchRequest searchRequest = new SearchRequest().indices(NOTIFICATION_INDEX_ALIAS_NAME).source(new SearchSourceBuilder().from(start).size(limit).version(true).query(this.getQuery(userName, roles, organisations)).sort("date", SortOrder.DESC));
        SearchResponse searchResponse = this.oss.search(searchRequest, RequestOptions.DEFAULT);
        Notifications notifications = new Notifications(start, limit);
        notifications.setTotalCountFromSource(searchResponse.getHits().getTotalHits().value());
        for (SearchHit hit : searchResponse.getHits().getHits()) {
            String notificationId = hit.getId();
            try {
                Notification notification = this.createNotificationFromOpenSearchSource(hit.getVersion(), hit.getSourceAsString());
                notifications.addData(notification.toGetDto(userName));
            }
            catch (Exception ex) {
                LOGGER.warn("Could not create a Notification object from a OpenSearch response hit. Please fix the notification with the ID '{}' in the index 'bpc-notification'.", (Object)notificationId);
                notifications.addAsIdWithError(notificationId);
            }
        }
        return notifications;
    }

    public List<Notification> loadNotifications(List<String> notificationIds) throws OpenSearchRelatedException, NotificationException {
        ArrayList<Notification> notifications = new ArrayList<Notification>();
        MultiGetRequest multiGetRequest = new MultiGetRequest();
        for (String id : notificationIds) {
            multiGetRequest.add(NOTIFICATION_INDEX_ALIAS_NAME, id);
        }
        MultiGetResponse response = this.oss.mget(multiGetRequest, RequestOptions.DEFAULT);
        for (MultiGetItemResponse item : response) {
            GetResponse respItem;
            if (item.isFailed() || (respItem = item.getResponse()) == null || !respItem.isExists()) continue;
            Notification notification = this.createNotificationFromOpenSearchSource(respItem.getVersion(), respItem.getSourceAsString());
            notifications.add(notification);
        }
        return notifications;
    }

    public List<Notification> loadNotificationsByTopic(String topic) throws OpenSearchRelatedException, OpenSearchIndexNotFoundException {
        LOGGER.info("loadNotificationsByTopic topic={}", (Object)topic);
        ArrayList<Notification> notifications = new ArrayList<Notification>();
        SearchRequest searchRequest = new SearchRequest().indices(NOTIFICATION_INDEX_ALIAS_NAME).source(new SearchSourceBuilder().version(true).query(QueryBuilders.termQuery("topic", topic)).sort("date", SortOrder.DESC));
        this.oss.search(searchRequest, searchHit -> {
            String notificationId = searchHit.getId();
            try {
                Notification notification = this.createNotificationFromOpenSearchSource(searchHit.getVersion(), searchHit.getSourceAsString());
                notifications.add(notification);
            }
            catch (Exception ex) {
                LOGGER.warn("Could not create a Notification object from a OpenSearch response hit. Please fix the notification with the ID '{}' in the index 'bpc-notification'.", (Object)notificationId);
            }
        });
        return notifications;
    }

    private QueryBuilder getQuery(String userName, List<Role> roles, List<Organisation> organisations) {
        BoolQueryBuilder userAccessQuery = QueryBuilders.boolQuery();
        if (!StringUtil.isNullOrEmpty(userName)) {
            userAccessQuery.should(QueryBuilders.boolQuery().must(QueryBuilders.termQuery("recipientsType", "user")).must(QueryBuilders.termQuery("recipients", userName)));
        }
        if (roles != null && !roles.isEmpty()) {
            BoolQueryBuilder oneOfTheRolesQuery = QueryBuilders.boolQuery();
            for (Role role : roles) {
                oneOfTheRolesQuery.should(QueryBuilders.termQuery("recipients", role.getName()));
            }
            userAccessQuery.should(QueryBuilders.boolQuery().must(QueryBuilders.termQuery("recipientsType", "role")).must(oneOfTheRolesQuery));
        }
        if (organisations != null && !organisations.isEmpty()) {
            BoolQueryBuilder oneOfTheOrganisationsQuery = QueryBuilders.boolQuery();
            for (Organisation organisation : organisations) {
                oneOfTheOrganisationsQuery.should(QueryBuilders.termQuery("recipients", organisation.getName()));
            }
            userAccessQuery.should(QueryBuilders.boolQuery().must(QueryBuilders.termQuery("recipientsType", "organisation")).must(oneOfTheOrganisationsQuery));
        }
        ExistsQueryBuilder expiryDateExists = QueryBuilders.existsQuery("validUntil");
        RangeQueryBuilder expiryInFuture = QueryBuilders.rangeQuery("validUntil").gt(Instant.now().toString());
        BoolQueryBuilder expiryFilter = QueryBuilders.boolQuery().should(QueryBuilders.boolQuery().mustNot(expiryDateExists)).should(expiryInFuture).minimumShouldMatch(1);
        return QueryBuilders.boolQuery().must(userAccessQuery).filter(expiryFilter);
    }

    private Notification createNotificationFromOpenSearchSource(long version, String sourceAsString) throws NotificationException {
        try {
            NotificationData osData = JsonUtil.getInstance().convertJsonStringToPojo(sourceAsString, NotificationData.class);
            return new Notification.Builder(osData).version(version).build();
        }
        catch (Exception ex) {
            throw new NotificationException((ErrorCode)CoreErrorCode.NOTIFICATION_FAILED, "Failed to convert a notification object to a JSON string.", ex);
        }
    }
}

