/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.routing;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.routing.ShardRouting;
import org.opensearch.cluster.routing.ShardRoutingState;
import org.opensearch.common.Nullable;
import org.opensearch.common.annotation.PublicApi;
import org.opensearch.common.collect.Tuple;
import org.opensearch.core.index.Index;
import org.opensearch.core.index.shard.ShardId;

@PublicApi(since="1.0.0")
public class RoutingNode
implements Iterable<ShardRouting> {
    private final String nodeId;
    private final DiscoveryNode node;
    private final BucketedShards shards;
    private final RelocatingShardsBucket relocatingShardsBucket;
    private final LinkedHashSet<ShardRouting> initializingShards;
    private final HashMap<Index, LinkedHashSet<ShardRouting>> shardsByIndex;

    public RoutingNode(String nodeId, DiscoveryNode node, ShardRouting ... shardRoutings) {
        this.nodeId = nodeId;
        this.node = node;
        LinkedHashMap<ShardId, ShardRouting> primaryShards = new LinkedHashMap<ShardId, ShardRouting>();
        LinkedHashMap<ShardId, ShardRouting> replicaShards = new LinkedHashMap<ShardId, ShardRouting>();
        this.shards = new BucketedShards(primaryShards, replicaShards);
        this.relocatingShardsBucket = new RelocatingShardsBucket();
        this.initializingShards = new LinkedHashSet();
        this.shardsByIndex = new LinkedHashMap<Index, LinkedHashSet<ShardRouting>>();
        for (ShardRouting shardRouting : shardRoutings) {
            if (shardRouting.initializing()) {
                this.initializingShards.add(shardRouting);
            } else if (shardRouting.relocating()) {
                this.relocatingShardsBucket.add(shardRouting);
            }
            this.shardsByIndex.computeIfAbsent(shardRouting.index(), k -> new LinkedHashSet()).add(shardRouting);
            ShardRouting previousValue = shardRouting.primary() ? primaryShards.put(shardRouting.shardId(), shardRouting) : replicaShards.put(shardRouting.shardId(), shardRouting);
            if (previousValue == null) continue;
            throw new IllegalArgumentException("Cannot have two different shards with same shard id " + String.valueOf(shardRouting.shardId()) + " on same node ");
        }
        assert (this.invariant());
    }

    @Override
    public Iterator<ShardRouting> iterator() {
        return this.shards.iterator();
    }

    public DiscoveryNode node() {
        return this.node;
    }

    @Nullable
    public ShardRouting getByShardId(ShardId id) {
        return this.shards.get(id);
    }

    public String nodeId() {
        return this.nodeId;
    }

    public int size() {
        return this.shards.size();
    }

    public Collection<ShardRouting> getInitializingShards() {
        return this.initializingShards;
    }

    void add(ShardRouting shard) {
        assert (this.invariant());
        if (this.shards.put(shard) != null) {
            throw new IllegalStateException("Trying to add a shard " + String.valueOf(shard.shardId()) + " to a node [" + this.nodeId + "] where it already exists. current [" + String.valueOf(this.shards.get(shard.shardId())) + "]. new [" + String.valueOf(shard) + "]");
        }
        if (shard.initializing()) {
            this.initializingShards.add(shard);
        } else if (shard.relocating()) {
            this.relocatingShardsBucket.add(shard);
        }
        this.shardsByIndex.computeIfAbsent(shard.index(), k -> new LinkedHashSet()).add(shard);
        assert (this.invariant());
    }

    void update(ShardRouting oldShard, ShardRouting newShard) {
        assert (this.invariant());
        if (!this.shards.containsKey(oldShard.shardId())) {
            return;
        }
        ShardRouting previousValue = this.shards.put(newShard.shardId(), newShard);
        assert (previousValue == oldShard) : "expected shard " + String.valueOf(previousValue) + " but was " + String.valueOf(oldShard);
        if (oldShard.initializing()) {
            boolean exist = this.initializingShards.remove(oldShard);
            assert (exist) : "expected shard " + String.valueOf(oldShard) + " to exist in initializingShards";
        } else if (oldShard.relocating()) {
            boolean exist = this.relocatingShardsBucket.remove(oldShard);
            assert (exist) : "expected shard " + String.valueOf(oldShard) + " to exist in relocatingShards";
        }
        this.shardsByIndex.get(oldShard.index()).remove(oldShard);
        if (this.shardsByIndex.get(oldShard.index()).isEmpty()) {
            this.shardsByIndex.remove(oldShard.index());
        }
        if (newShard.initializing()) {
            this.initializingShards.add(newShard);
        } else if (newShard.relocating()) {
            this.relocatingShardsBucket.add(newShard);
        }
        this.shardsByIndex.computeIfAbsent(newShard.index(), k -> new LinkedHashSet()).add(newShard);
        assert (this.invariant());
    }

    void remove(ShardRouting shard) {
        assert (this.invariant());
        ShardRouting previousValue = this.shards.remove(shard.shardId());
        assert (previousValue == shard) : "expected shard " + String.valueOf(previousValue) + " but was " + String.valueOf(shard);
        if (shard.initializing()) {
            boolean exist = this.initializingShards.remove(shard);
            assert (exist) : "expected shard " + String.valueOf(shard) + " to exist in initializingShards";
        } else if (shard.relocating()) {
            boolean exist = this.relocatingShardsBucket.remove(shard);
            assert (exist) : "expected shard " + String.valueOf(shard) + " to exist in relocatingShards";
        }
        this.shardsByIndex.get(shard.index()).remove(shard);
        if (this.shardsByIndex.get(shard.index()).isEmpty()) {
            this.shardsByIndex.remove(shard.index());
        }
        assert (this.invariant());
    }

    public int numberOfShardsWithState(ShardRoutingState ... states) {
        if (states.length == 1) {
            if (states[0] == ShardRoutingState.INITIALIZING) {
                return this.initializingShards.size();
            }
            if (states[0] == ShardRoutingState.RELOCATING) {
                return this.relocatingShardsBucket.size();
            }
        }
        int count = 0;
        for (ShardRouting shardEntry : this) {
            for (ShardRoutingState state : states) {
                if (shardEntry.state() != state) continue;
                ++count;
            }
        }
        return count;
    }

    public List<ShardRouting> shardsWithState(ShardRoutingState ... states) {
        if (states.length == 1) {
            if (states[0] == ShardRoutingState.INITIALIZING) {
                return new ArrayList<ShardRouting>(this.initializingShards);
            }
            if (states[0] == ShardRoutingState.RELOCATING) {
                return this.relocatingShardsBucket.getRelocatingShardsList();
            }
        }
        ArrayList<ShardRouting> shards = new ArrayList<ShardRouting>();
        for (ShardRouting shardEntry : this) {
            for (ShardRoutingState state : states) {
                if (shardEntry.state() != state) continue;
                shards.add(shardEntry);
            }
        }
        return shards;
    }

    public List<ShardRouting> shardsWithState(String index, ShardRoutingState ... states) {
        ArrayList<ShardRouting> shards = new ArrayList<ShardRouting>();
        if (states.length == 1) {
            if (states[0] == ShardRoutingState.INITIALIZING) {
                for (ShardRouting shardEntry : this.initializingShards) {
                    if (!shardEntry.getIndexName().equals(index)) continue;
                    shards.add(shardEntry);
                }
                return shards;
            }
            if (states[0] == ShardRoutingState.RELOCATING) {
                for (ShardRouting shardEntry : this.relocatingShardsBucket.getRelocatingShards()) {
                    if (!shardEntry.getIndexName().equals(index)) continue;
                    shards.add(shardEntry);
                }
                return shards;
            }
        }
        for (ShardRouting shardEntry : this) {
            if (!shardEntry.getIndexName().equals(index)) continue;
            for (ShardRoutingState state : states) {
                if (shardEntry.state() != state) continue;
                shards.add(shardEntry);
            }
        }
        return shards;
    }

    public int numberOfOwningShards() {
        return this.shards.size() - this.relocatingShardsBucket.size();
    }

    public int numberOfOwningPrimaryShards() {
        return this.shards.numberOfPrimaryShards() - this.relocatingShardsBucket.primarySize();
    }

    public int numberOfOwningShardsForIndex(Index index) {
        LinkedHashSet<ShardRouting> shardRoutings = this.shardsByIndex.get(index);
        if (shardRoutings == null) {
            return 0;
        }
        return Math.toIntExact(shardRoutings.stream().filter(sr -> !sr.relocating()).count());
    }

    public int numberOfOwningPrimaryShardsForIndex(Index index) {
        LinkedHashSet<ShardRouting> shardRoutings = this.shardsByIndex.get(index);
        if (shardRoutings == null) {
            return 0;
        }
        return Math.toIntExact(shardRoutings.stream().filter(sr -> !sr.relocating()).filter(ShardRouting::primary).count());
    }

    public String prettyPrint() {
        StringBuilder sb = new StringBuilder();
        sb.append("-----node_id[").append(this.nodeId).append("][").append(this.node == null ? "X" : "V").append("]\n");
        for (ShardRouting entry : this.shards) {
            sb.append("--------").append(entry.shortSummary()).append('\n');
        }
        return sb.toString();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("routingNode ([");
        sb.append(this.node.getName());
        sb.append("][");
        sb.append(this.node.getId());
        sb.append("][");
        sb.append(this.node.getHostName());
        sb.append("][");
        sb.append(this.node.getHostAddress());
        sb.append("], [");
        sb.append(this.shards.size());
        sb.append(" assigned shards])");
        return sb.toString();
    }

    public List<ShardRouting> copyShards() {
        ArrayList<ShardRouting> result = new ArrayList<ShardRouting>();
        this.shards.forEach(result::add);
        return result;
    }

    public boolean isEmpty() {
        return this.shards.isEmpty();
    }

    private boolean invariant() {
        Collection shardRoutingsInitializing = StreamSupport.stream(this.shards.spliterator(), false).filter(ShardRouting::initializing).collect(Collectors.toList());
        assert (this.initializingShards.size() == shardRoutingsInitializing.size());
        assert (this.initializingShards.containsAll(shardRoutingsInitializing));
        Collection shardRoutingsRelocating = StreamSupport.stream(this.shards.spliterator(), false).filter(ShardRouting::relocating).collect(Collectors.toList());
        assert (this.relocatingShardsBucket.getRelocatingShards().size() == shardRoutingsRelocating.size());
        assert (this.relocatingShardsBucket.getRelocatingShards().containsAll(shardRoutingsRelocating));
        Collection primaryShardRoutingsRelocating = StreamSupport.stream(this.shards.spliterator(), false).filter(ShardRouting::relocating).filter(ShardRouting::primary).collect(Collectors.toList());
        assert (this.relocatingShardsBucket.getRelocatingPrimaryShards().size() == primaryShardRoutingsRelocating.size());
        assert (this.relocatingShardsBucket.getRelocatingPrimaryShards().containsAll(primaryShardRoutingsRelocating));
        assert (this.relocatingShardsBucket.invariant());
        Map shardRoutingsByIndex = StreamSupport.stream(this.shards.spliterator(), false).collect(Collectors.groupingBy(ShardRouting::index, Collectors.toSet()));
        assert (shardRoutingsByIndex.equals(this.shardsByIndex));
        return true;
    }

    static class BucketedShards
    implements Iterable<ShardRouting> {
        private final Tuple<LinkedHashMap<ShardId, ShardRouting>, LinkedHashMap<ShardId, ShardRouting>> shardTuple;

        BucketedShards(LinkedHashMap<ShardId, ShardRouting> primaryShards, LinkedHashMap<ShardId, ShardRouting> replicaShards) {
            this.shardTuple = new Tuple<LinkedHashMap<ShardId, ShardRouting>, LinkedHashMap<ShardId, ShardRouting>>(primaryShards, replicaShards);
        }

        public boolean isEmpty() {
            return this.shardTuple.v1().isEmpty() && this.shardTuple.v2().isEmpty();
        }

        public int size() {
            return this.shardTuple.v1().size() + this.shardTuple.v2().size();
        }

        public boolean containsKey(ShardId shardId) {
            return this.shardTuple.v1().containsKey(shardId) || this.shardTuple.v2().containsKey(shardId);
        }

        public ShardRouting get(ShardId shardId) {
            if (this.shardTuple.v1().containsKey(shardId)) {
                return this.shardTuple.v1().get(shardId);
            }
            return this.shardTuple.v2().get(shardId);
        }

        public ShardRouting put(ShardRouting shardRouting) {
            return this.put(shardRouting.shardId(), shardRouting);
        }

        public ShardRouting put(ShardId shardId, ShardRouting shardRouting) {
            ShardRouting ret;
            if (shardRouting.primary()) {
                ret = this.shardTuple.v1().put(shardId, shardRouting);
                if (this.shardTuple.v2().containsKey(shardId)) {
                    ret = (ShardRouting)this.shardTuple.v2().remove(shardId);
                }
            } else {
                ret = this.shardTuple.v2().put(shardId, shardRouting);
                if (this.shardTuple.v1().containsKey(shardId)) {
                    ret = (ShardRouting)this.shardTuple.v1().remove(shardId);
                }
            }
            return ret;
        }

        public ShardRouting remove(ShardId shardId) {
            if (this.shardTuple.v1().containsKey(shardId)) {
                return (ShardRouting)this.shardTuple.v1().remove(shardId);
            }
            return (ShardRouting)this.shardTuple.v2().remove(shardId);
        }

        @Override
        public Iterator<ShardRouting> iterator() {
            return Stream.concat(Collections.unmodifiableCollection(this.shardTuple.v1().values()).stream(), Collections.unmodifiableCollection(this.shardTuple.v2().values()).stream()).iterator();
        }

        public int numberOfPrimaryShards() {
            return this.shardTuple.v1().size();
        }
    }

    static class RelocatingShardsBucket {
        private final LinkedHashSet<ShardRouting> relocatingShards = new LinkedHashSet();
        private final LinkedHashSet<ShardRouting> relocatingPrimaryShards = new LinkedHashSet();

        RelocatingShardsBucket() {
        }

        public boolean add(ShardRouting shard) {
            boolean res = this.relocatingShards.add(shard);
            if (shard.primary()) {
                this.relocatingPrimaryShards.add(shard);
            }
            return res;
        }

        public boolean remove(ShardRouting shard) {
            boolean res = this.relocatingShards.remove(shard);
            this.relocatingPrimaryShards.remove(shard);
            return res;
        }

        public int size() {
            return this.relocatingShards.size();
        }

        public int primarySize() {
            return this.relocatingPrimaryShards.size();
        }

        public Set<ShardRouting> getRelocatingShards() {
            return Collections.unmodifiableSet(this.relocatingShards);
        }

        public Set<ShardRouting> getRelocatingPrimaryShards() {
            return Collections.unmodifiableSet(this.relocatingPrimaryShards);
        }

        public List<ShardRouting> getRelocatingShardsList() {
            return new ArrayList<ShardRouting>(this.relocatingShards);
        }

        public boolean invariant() {
            assert (this.relocatingShards.containsAll(this.relocatingPrimaryShards));
            assert (this.relocatingPrimaryShards.stream().allMatch(ShardRouting::primary));
            assert ((long)this.relocatingPrimaryShards.size() == this.relocatingShards.stream().filter(ShardRouting::primary).count());
            return true;
        }
    }
}

