/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.codec.fuzzy;

import java.io.Closeable;
import java.io.IOException;
import java.util.Iterator;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.RamUsageEstimator;
import org.opensearch.common.CheckedSupplier;
import org.opensearch.common.util.io.IOUtils;
import org.opensearch.core.Assertions;
import org.opensearch.index.codec.fuzzy.AbstractFuzzySet;
import org.opensearch.index.codec.fuzzy.FuzzySet;
import org.opensearch.index.codec.fuzzy.LongArrayBackedBitSet;

public class BloomFilter
extends AbstractFuzzySet {
    private static final Logger logger = LogManager.getLogger(BloomFilter.class);
    static final int[] usableBitSetSizes = new int[26];
    private final LongArrayBackedBitSet bitset;
    private final int setSize;
    private final int hashCount;

    BloomFilter(long maxDocs, double maxFpp, CheckedSupplier<Iterator<BytesRef>, IOException> fieldIteratorProvider) throws IOException {
        int setSize = (int)Math.ceil((double)maxDocs * Math.log(maxFpp) / Math.log(1.0 / Math.pow(2.0, Math.log(2.0))));
        setSize = BloomFilter.getNearestSetSize(setSize < 0x3FFFFFFF ? 2 * setSize : Integer.MAX_VALUE);
        int optimalK = (int)Math.round((double)setSize / (double)maxDocs * Math.log(2.0));
        this.bitset = new LongArrayBackedBitSet(setSize);
        this.setSize = setSize;
        this.hashCount = optimalK;
        this.addAll(fieldIteratorProvider);
        if (Assertions.ENABLED) {
            this.assertAllElementsExist(fieldIteratorProvider);
        }
        logger.debug("Bloom filter created with fpp: {}, setSize: {}, hashCount: {}", (Object)maxFpp, (Object)setSize, (Object)this.hashCount);
    }

    BloomFilter(IndexInput in) throws IOException {
        this.hashCount = in.readInt();
        this.setSize = in.readInt();
        this.bitset = new LongArrayBackedBitSet(in);
    }

    @Override
    public void writeTo(DataOutput out) throws IOException {
        out.writeInt(this.hashCount);
        out.writeInt(this.setSize);
        this.bitset.writeTo(out);
    }

    private static int getNearestSetSize(int maxNumberOfBits) {
        assert (maxNumberOfBits > 0) : "Provided size estimate for bloom filter is illegal (<=0) : " + maxNumberOfBits;
        int result = usableBitSetSizes[0];
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            if (usableBitSetSizes[i] > maxNumberOfBits) continue;
            result = usableBitSetSizes[i];
        }
        return result;
    }

    @Override
    public FuzzySet.SetType setType() {
        return FuzzySet.SetType.BLOOM_FILTER_V1;
    }

    @Override
    public FuzzySet.Result containsHash(long hash) {
        int msb = (int)(hash >>> 32);
        int lsb = (int)hash;
        for (int i = 0; i < this.hashCount; ++i) {
            int bloomPos = lsb + i * msb;
            if (this.mayContainValue(bloomPos)) continue;
            return FuzzySet.Result.NO;
        }
        return FuzzySet.Result.MAYBE;
    }

    @Override
    protected void add(BytesRef value) {
        long hash = this.generateKey(value);
        int msb = (int)(hash >>> 32);
        int lsb = (int)hash;
        for (int i = 0; i < this.hashCount; ++i) {
            int bloomPos = lsb + i * msb & this.setSize;
            this.bitset.set(bloomPos);
        }
    }

    @Override
    public boolean isSaturated() {
        long numBitsSet = this.bitset.cardinality();
        return (float)numBitsSet / (float)this.setSize > 0.9f;
    }

    @Override
    public long ramBytesUsed() {
        return RamUsageEstimator.sizeOf(this.bitset.ramBytesUsed());
    }

    private boolean mayContainValue(int aHash) {
        int pos = aHash & this.setSize;
        return this.bitset.get(pos);
    }

    @Override
    public void close() throws IOException {
        IOUtils.close((Closeable)this.bitset);
    }

    static {
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            BloomFilter.usableBitSetSizes[i] = (1 << i + 6) - 1;
        }
    }
}

